package ru.yandex.webmaster3.internal.user.domain.verification;

import org.joda.time.DateTime;
import org.joda.time.base.AbstractInstant;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.UinUtil;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationStatus;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;

/**
 * @author avhaliullin
 */
public class DomainVerificationInfoResponse implements ActionResponse.NormalResponse {
    private final WebmasterHostId hostId;
    private final VerificationStatus verificationStatus;
    private final VerificationType verificationType;
    private final String verificationUin;
    private final DateTime lastVerificationAttemptTime;
    private final DateTime nextVerificationAttemptTime;
    private final VerificationFailInfo lastVerificationFailInfo;
    private final DateTime verifiedDate;

    public DomainVerificationInfoResponse(WebmasterHostId hostId, VerificationStatus verificationStatus, VerificationType verificationType,
                                          String verificationUin, AbstractInstant nextVerificationAttemptTime, AbstractInstant lastVerificationAttemptTime,
                                          VerificationFailInfo lastVerificationFailInfo, AbstractInstant verifiedDate) {
        this.hostId = hostId;
        this.verificationStatus = verificationStatus;
        this.verificationType = verificationType;
        this.verificationUin = verificationUin;
        this.nextVerificationAttemptTime = toDateTime(nextVerificationAttemptTime);
        this.lastVerificationAttemptTime = toDateTime(lastVerificationAttemptTime);
        this.lastVerificationFailInfo = lastVerificationFailInfo;
        this.verifiedDate = toDateTime(verifiedDate);
    }

    @Description("подтверждаемый хост")
    public WebmasterHostId getHostId() {
        return hostId;
    }

    @Description("null, если подтверждение прав не производилось")
    public VerificationStatus getVerificationStatus() {
        return verificationStatus;
    }

    public VerificationType getVerificationType() {
        return verificationType;
    }

    @Description("код для подтверждения прав")
    public String getVerificationUin() {
        return verificationUin;
    }

    @Description("дата следующей проверки")
    public DateTime getNextVerificationAttemptTime() {
        return nextVerificationAttemptTime;
    }

    @Description("дата последней проверки")
    public DateTime getLastVerificationAttemptTime() {
        return lastVerificationAttemptTime;
    }

    @Description("результат последней неудачной проверки")
    public VerificationFailInfo getLastVerificationFailInfo() {
        return lastVerificationFailInfo;
    }

    @Description("дата подтверждения прав (если подтверждены)")
    public DateTime getVerifiedDate() {
        return verifiedDate;
    }

    private static DateTime toDateTime(AbstractInstant instant) {
        if (instant == null) {
            return null;
        }
        return instant.toDateTime();
    }

    static DomainVerificationInfoResponse createVerified(UserVerifiedHost userVerifiedHost) {
        return new DomainVerificationInfoResponse(userVerifiedHost.getWebmasterHostId(), VerificationStatus.VERIFIED, userVerifiedHost.getVerificationType(),
                UinUtil.getUinString(userVerifiedHost.getVerificationUin()), null, null, null,
                userVerifiedHost.getVerificationDate()
        );
    }

    static DomainVerificationInfoResponse createNeverVerified(UserHostVerificationInfo verificationInfo) {
        return new DomainVerificationInfoResponse(verificationInfo.getHostId(), null, null,
                UinUtil.getUinString(verificationInfo.getVerificationUin()), null, null, null, null);
    }

    static DomainVerificationInfoResponse createNotVerified(UserHostVerificationInfo verificationInfo) {
        return new DomainVerificationInfoResponse(verificationInfo.getHostId(), verificationInfo.getVerificationStatus(),
                verificationInfo.getVerificationType(),
                UinUtil.getUinString(verificationInfo.getVerificationUin()),
                verificationInfo.getNextAttempt(), verificationInfo.getLastAttempt(), verificationInfo.getVerificationFailInfo(),
                null);
    }

    static DomainVerificationInfoResponse createExpired(UserHostVerificationInfo verificationInfo) {
        return new DomainVerificationInfoResponse(verificationInfo.getHostId(), VerificationStatus.INTERNAL_ERROR,
                verificationInfo.getVerificationType(),
                UinUtil.getUinString(verificationInfo.getVerificationUin()),
                null, verificationInfo.getLastAttempt(), verificationInfo.getVerificationFailInfo(),
                null);
    }
}

