package ru.yandex.webmaster3.internal.user.host.verification;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.UserAddedHost;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.host.verification.VerificationTypeScope;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.internal.common.InternalAction;
import ru.yandex.webmaster3.internal.common.security.ActionInternalGrant;
import ru.yandex.webmaster3.internal.common.security.InternalGrant;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;

/**
 * Created by leonidrom on 13/06/2017.
 */
@Description("Запустить процедуру подтверждения прав на хост")
@WriteAction
@ActionInternalGrant(InternalGrant.HOST_VERIFICATION)
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class VerifyHostAction extends InternalAction<VerifyHostRequest, VerifyHostResponse> {

    private final UserHostVerificationYDao userHostVerificationYDao;
    private final HostVerifierService hostVerifierService;
    private final UserHostsService userHostsService;

    @Override
    public VerifyHostResponse process(VerifyHostRequest request) throws WebmasterException {
        if (!request.getVerificationType().visibleIn(VerificationTypeScope.INTERNAL)) {
            return new VerifyHostResponse.VerificationNotApplicable();
        }
        UserAddedHost userAddedHost = request.getUserAddedHost();
        if (userAddedHost.isVerified()) {
            boolean isPDD = false;
            UserVerifiedHost userVerifiedHost = userHostsService.getVerifiedHost(request.getWebmasterUser(), request.getHostId());
            if (userVerifiedHost != null) {
                isPDD = userVerifiedHost.getVerificationType() == VerificationType.PDD;
            }

            // ПДД закрыто, поэтому разрешим перезаписать это подтверждение
            if (!isPDD) {
                return new VerifyHostResponse.AlreadyVerifiedError();
            }
        }

        if (!hostVerifierService.isApplicable(request.getHostId(), request.getUserId(), request.getVerificationType())) {
            return new VerifyHostResponse.VerificationNotApplicable();
        }

        UserHostVerificationInfo verificationInfo = userHostVerificationYDao.getLatestRecord(request.getUserId(), request.getHostId());
        if (HostVerifierService.isVerificationInProgress(verificationInfo)) {
            return new VerifyHostResponse.VerificationIsInProgressError();
        }

        boolean launched = hostVerifierService.initiateVerification(
                verificationInfo,
                VerificationCausedBy.INTERNAL_API,
                request.getHostId(),
                request.getUserId(),
                request.getVerificationType()
        );
        return launched
                ? new VerifyHostResponse.NormalResponse()
                : new VerifyHostResponse.VerificationNotApplicable();
    }

}
