package ru.yandex.webmaster3.internal.user.takeout;

import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.internal.common.InternalAction;
import ru.yandex.webmaster3.internal.common.security.InternalGrant;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequestStatus;
import ru.yandex.webmaster3.storage.user.dao.InitializedUsersYDao;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutService;

import javax.annotation.PostConstruct;

import static ru.yandex.webmaster3.core.util.json.JsonMapping.OM;

/**
 * @author leonidrom
 */
@Component("/1/takeout/status")
@ReadAction
@Slf4j
public class UserTakeoutDeleteStatusAction  extends InternalAction<UserTakeoutDeleteStatusRequest, UserTakeoutDeleteStatusResponse> {
    private static final ObjectNode ID_SLUG_NODE = OM.createObjectNode();

    private final InitializedUsersYDao initializedUsersYDao;
    private final UserTakeoutService userTakeoutService;

    @PostConstruct
    public void init() {
        ID_SLUG_NODE.put("id", "1");
        ID_SLUG_NODE.put("slug", "general");
    }

    @Autowired
    public UserTakeoutDeleteStatusAction(InitializedUsersYDao initializedUsersYDao, UserTakeoutService userTakeoutService) {
        super(InternalGrant.USER_TAKEOUT);
        this.initializedUsersYDao = initializedUsersYDao;
        this.userTakeoutService = userTakeoutService;
    }

    @Override
    public UserTakeoutDeleteStatusResponse process(UserTakeoutDeleteStatusRequest request) throws WebmasterException {
        try {
            return doProcess(request);
        } catch (Exception e) {
            // нужно для того, чтобы отдать ошибку в формате, ожидаемом takeout
            log.error("Error while processing action", e);
            return new UserTakeoutDeleteStatusResponse.ErrorResponse("internal", "Error while processing action");
        }
    }

    private UserTakeoutDeleteStatusResponse doProcess(UserTakeoutDeleteStatusRequest request) {
        log.info("Request: " + request.getRequestId());
        long userId = request.getUserId();

        var takeoutReq = userTakeoutService.getRequest(userId);
        var userInfo = initializedUsersYDao.getUserInfo(userId);

        if (takeoutReq == null || takeoutReq.getStatus() == UserTakeoutRequestStatus.DONE) {
            var lastUpdate = takeoutReq != null ? takeoutReq.getLastUpdate().toInstant() : null;
            if (userInfo == null) {
                return createNoDataResponse(lastUpdate);
            } else {
                return createReadyToDeleteResponse(lastUpdate);
            }
        }

        var status = takeoutReq.getStatus();
        switch (status) {
            case IN_PROGRESS -> {
                return createInProgressResponse();
            }

            case FAILED -> {
                return new UserTakeoutDeleteStatusResponse.ErrorResponse("internal", "Error deleting user data");
            }
        }

        return createReadyToDeleteResponse(null);
    }

    public UserTakeoutDeleteStatusResponse.NormalResponse createNoDataResponse(Instant lastDeleteDate) {
        ObjectNode n = ID_SLUG_NODE.deepCopy();
        n.put("state", "empty");
        if (lastDeleteDate != null) {
            n.put("update_date", new DateTime(lastDeleteDate).toString());
        }

        ArrayNode data = OM.createArrayNode();
        data.add(n);

        return new UserTakeoutDeleteStatusResponse.NormalResponse(data);
    }

    public UserTakeoutDeleteStatusResponse.NormalResponse createReadyToDeleteResponse(Instant lastDeleteDate) {
        ObjectNode n = ID_SLUG_NODE.deepCopy();
        n.put("state", "ready_to_delete");
        if (lastDeleteDate != null) {
            n.put("update_date", new DateTime(lastDeleteDate).toString());
        }

        ArrayNode data = OM.createArrayNode();
        data.add(n);

        return new UserTakeoutDeleteStatusResponse.NormalResponse(data);
    }

    public UserTakeoutDeleteStatusResponse.NormalResponse createInProgressResponse() {
        ObjectNode n = ID_SLUG_NODE.deepCopy();
        n.put("state", "delete_in_progress");
        ArrayNode data = OM.createArrayNode();
        data.add(n);

        return new UserTakeoutDeleteStatusResponse.NormalResponse(data);
    }
}