package ru.yandex.webmaster3.internal.user.takeout;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.internal.common.InternalAction;
import ru.yandex.webmaster3.internal.common.security.InternalGrant;
import ru.yandex.webmaster3.storage.postpone.PostponeActionYDao;
import ru.yandex.webmaster3.storage.postpone.PostponeOperationType;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequest;
import ru.yandex.webmaster3.core.user.takeout.UserTakeoutRequestType;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutQueryData;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutService;

/**
 * @author leonidrom
 * <p>
 * Ручку нужно явно делать доступной. Подробнее см. Main.java
 */
@Slf4j
@WriteAction
@Description("Создает запрос на асинхронную выгрузку пользоательских данных")
public class UserTakeoutNewAction extends InternalAction<UserTakeoutNewRequest, UserTakeoutNewResponse> {
    private final UserTakeoutService userTakeoutService;
    private final PostponeActionYDao postponeActionYDao;

    @Autowired
    public UserTakeoutNewAction(UserTakeoutService userTakeoutService, PostponeActionYDao postponeActionYDao) {
        super(InternalGrant.USER_TAKEOUT);
        this.userTakeoutService = userTakeoutService;
        this.postponeActionYDao = postponeActionYDao;
    }

    @Override
    public UserTakeoutNewResponse process(UserTakeoutNewRequest request) throws WebmasterException {
        log.info("Takeout request for uid {}", request.getUid());
        UserTakeoutRequest req = userTakeoutService.createRequest(request.getUid(), UserTakeoutRequestType.GET);
        log.info("Job id for uid {}: {}", request.getUid(), req.getRequestId());
        postponeActionYDao.insert(DateTime.now().plus(UserTakeoutService.GET_RETRY_PERIOD),
                PostponeOperationType.USER_TAKEOUT_GET,
                JsonMapping.writeValueAsString(new UserTakeoutQueryData(request.getUid(), req.getRequestId())));
        return new UserTakeoutNewResponse(req.getRequestId());
    }
}
