package ru.yandex.webmaster3.monitoring.addurl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableLong;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.webmaster3.core.addurl.UrlForRecrawl;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.*;
import ru.yandex.webmaster3.core.util.DailyQuotaUtil;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.addurl.AddUrlOwnerRequestsYDao;
import ru.yandex.webmaster3.storage.addurl.AddUrlRequestsService;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataState;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataType;
import ru.yandex.webmaster3.storage.monitoring.common.dao.MonitoringDataStateYDao;
import ru.yandex.webmaster3.storage.spam.DeepSpamHostFilter;
import ru.yandex.webmaster3.storage.spam.FastSpamHostFilter;

import java.util.*;

import static org.joda.time.DateTimeConstants.SECONDS_PER_MINUTE;

/**
 * Created by Oleg Bazdyrev on 12/05/2017.
 */
@Slf4j
public class AddRecrawlUrlMonitoringService {
    private static final String SECTION_LABEL_VALUE = "recrawl";
    private static final String NEW_REQUESTS_DATA_TYPE = "new_requests";
    private static final String IN_PROGRESS_REQUESTS_DATA_TYPE = "in_progress_requests";
    private static final String PROCESSED_REQUESTS_DATA_TYPE = "processed_requests";
    private static final String STALE_REQUESTS_DATA_TYPE = "stale_requests";

    // число владельцев, которые использовали всю свою квоту
    private static final String QUOTA_LIMIT_REACHED_DATA_TYPE = "quota_limit_reached";

    // суммарный объем использованный квоты среди владельцев, которые полностью выгребли свою квоту
    private static final String QUOTA_LIMIT_USED_DATA_TYPE = "quota_limit_used";

    private static final int AVERAGE_SENSORS_SIZE = 600;

    private static final List<Duration> BUCKETS = Arrays.asList(
            Duration.standardMinutes(10),
            Duration.standardMinutes(30),
            Duration.standardHours(1),
            Duration.standardHours(2),
            Duration.standardHours(4),
            Duration.standardHours(6),
            Duration.standardHours(12),
            Duration.standardDays(1)
    );

    private HandleCommonMetricsService handleCommonMetricsService;
    private AddUrlOwnerRequestsYDao addUrlOwnerRequestsYDao;
    private DeepSpamHostFilter deepSpamHostFilter;
    private FastSpamHostFilter fastSpamHostFilter;
    private MonitoringDataStateYDao monitoringDataStateYDao;
    private AddUrlRequestsService addUrlRequestsService;

    private boolean enabled;
    private long refreshIntervalSeconds;

    // важно чтобы эта таска не стартовала одновременно с UpdateUrlStatePeriodicTask
    @Scheduled(cron = "${webmaster3.monitoring.solomon.addurl.refresh-cron}")
    private void push() throws Exception {
        if (!enabled) {
            return;
        }

        log.info("Started gathering pending requests stats.");
        long now = System.currentTimeMillis();
        var lastRunDS = monitoringDataStateYDao.getValue(MonitoringDataType.LAST_ADDURL_MONITORING_RUN);
        long lastRunMillis = lastRunDS == null? now - refreshIntervalSeconds * 1000 : lastRunDS.getLastUpdate().getMillis();

        Map<SolomonKey, SolomonCounterImpl> sensorsAcc = new HashMap<>();
        SolomonHistogram<Duration> inProgressAgeHist = createHistogram(sensorsAcc);
        AddUrlRequestsStatistics stats = new AddUrlRequestsStatistics(lastRunMillis);

        // собираем статистику
        List<UrlForRecrawl> reqs = addUrlRequestsService.listUnprocessed(
                DateTime.now().minus(UrlForRecrawl.STALE_REQUEST_AGE).minusDays(1), // вычитываем с запасом, чтобы увидеть STALE запросы
                DateTime.now().minus(Duration.standardMinutes(5)) // не хотим вычитывать живые данные, чтобы не было конфликта транзакций
        );
        reqs.forEach(req -> {
            WebmasterHostId hostId = req.getHostId();
            boolean spam = fastSpamHostFilter.checkHost(hostId) || deepSpamHostFilter.checkHost(hostId);
            if (!spam) {
                stats.accept(now, req, inProgressAgeHist);
            }
        });

        log.info("Min new request: {}", stats.minNewRequest);
        log.info("Finished pending requests stats.");

        log.info("Started gathering quota stats.");
        DateTime lastRunDate = new DateTime(lastRunMillis);
        Set<String> owners = new HashSet<>();
        MutableLong requestsCount = new MutableLong();
        addUrlOwnerRequestsYDao.foreachRequest(req -> {
            WebmasterHostId ownerId = IdUtils.urlToHostId(req.getOwner());
            boolean spam = fastSpamHostFilter.checkHost(ownerId) || deepSpamHostFilter.checkHost(ownerId);
            if (spam || req.getAddDateTime().isBefore(lastRunDate)) {
                return;
            }

            owners.add(req.getOwner());
            requestsCount.increment();
        });

        log.info("Requests: {}, owners: {}", requestsCount, owners.size());

        // Мониторинг запускается раз в час, поэтому игнорируем проблему новых суток
        var quotaStats = getQuotaStats(owners, DateTime.now());

        log.info("Finished gathering quota stats.");

        // отправляем сенсоры
        List<SolomonSensor> sensors = new ArrayList<>();
        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, (now - stats.minNewRequestDate) / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, NEW_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.newRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, NEW_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, (now - stats.minInProgressRequestDate) / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.inProgressRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.processedRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, PROCESSED_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.staleRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, STALE_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, quotaStats.getLeft())
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.COUNT)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, QUOTA_LIMIT_REACHED_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, quotaStats.getRight())
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.COUNT)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, QUOTA_LIMIT_USED_DATA_TYPE));

        sensorsAcc.forEach((key, counter) -> {
            sensors.add(SolomonSensor.createAligned(key, now, SECONDS_PER_MINUTE, counter.getAsLong()));
        });

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);
        monitoringDataStateYDao.update(new MonitoringDataState(MonitoringDataType.LAST_ADDURL_MONITORING_RUN, "", DateTime.now()));
    }


    private static SolomonHistogram<Duration> createHistogram(Map<SolomonKey, SolomonCounterImpl> sensorsAcc) {
        return SolomonHistogramImpl.create(duration -> createCounter(sensorsAcc, duration), BUCKETS);
    }

    private static SolomonCounter createCounter(Map<SolomonKey, SolomonCounterImpl> sensorsAcc, Duration duration) {
        return sensorsAcc.computeIfAbsent(SolomonKey.create(
                SolomonKey.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                        .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                        .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE)
                        .withLabel(SolomonKey.LABEL_TIME_BUCKET, "<" + duration.getStandardMinutes() + "m"),
                ign -> SolomonCounterImpl.create(false)
        );
    }

    private Pair<Long, Long> getQuotaStats(Set<String> owners, DateTime quotaDate) {
        long quotaLimitReachedCount = 0;
        long totalQuotaUsed = 0;
        for (String owner : owners) {
            DailyQuotaUtil.QuotaUsage quota = addUrlRequestsService.getOwnerQuotaUsage(owner, quotaDate);
            if (quota.getQuotaRemain() <= 0) {
                quotaLimitReachedCount++;
                totalQuotaUsed += quota.getQuotaUsed();
            }
        }

        return Pair.of(quotaLimitReachedCount, totalQuotaUsed);
    }

    @Required
    public void setHandleCommonMetricsService(HandleCommonMetricsService handleCommonMetricsService) {
        this.handleCommonMetricsService = handleCommonMetricsService;
    }

    @Required
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    @Required
    public void setRefreshIntervalSeconds(long refreshIntervalSeconds) {
        this.refreshIntervalSeconds = refreshIntervalSeconds;
    }

    @Required
    public void setFastSpamHostFilter(FastSpamHostFilter fastSpamHostFilter) {
        this.fastSpamHostFilter = fastSpamHostFilter;
    }

    @Required
    public void setDeepSpamHostFilter(DeepSpamHostFilter deepSpamHostFilter) {
        this.deepSpamHostFilter = deepSpamHostFilter;
    }

    @Required
    public void setMonitoringDataStateYDao(MonitoringDataStateYDao monitoringDataStateYDao) {
        this.monitoringDataStateYDao = monitoringDataStateYDao;
    }

    @Required
    public void setAddUrlOwnerRequestsYDao(AddUrlOwnerRequestsYDao addUrlOwnerRequestsYDao) {
        this.addUrlOwnerRequestsYDao = addUrlOwnerRequestsYDao;
    }

    @Required
    public void setAddUrlRequestsService(AddUrlRequestsService addUrlRequestsService) {
        this.addUrlRequestsService = addUrlRequestsService;
    }

    @RequiredArgsConstructor
    private static class AddUrlRequestsStatistics {
        final long lastRunMillis;

        long minNewRequestDate = System.currentTimeMillis();
        long newRequestsCount = 0;
        long minInProgressRequestDate = System.currentTimeMillis();
        long inProgressRequestsCount = 0;
        long processedRequestsCount = 0;
        long staleRequestsCount = 0;
        UrlForRecrawl minNewRequest;

        void accept(long now, UrlForRecrawl urlForRecrawl, SolomonHistogram<Duration> inProgressAgeHist) {
            long processedTS = urlForRecrawl.getProcessedDate().getMillis();
            switch (urlForRecrawl.getState()) {
                case NEW:
                    minNewRequestDate = Math.min(processedTS, minNewRequestDate);
                    if (minNewRequestDate == processedTS) {
                        minNewRequest = urlForRecrawl;
                    }
                    newRequestsCount++;
                    break;

                case IN_PROGRESS:
                    minInProgressRequestDate = Math.min(processedTS, minInProgressRequestDate);
                    inProgressRequestsCount++;
                    var requestAge = Duration.millis(now - urlForRecrawl.getAddDate().getMillis());
                    inProgressAgeHist.update(requestAge);
                    break;

                case PROCESSED:
                    if (processedTS >= lastRunMillis) {
                        processedRequestsCount++;
                    }
                    break;

                case STALE:
                    if (processedTS >= lastRunMillis) {
                        staleRequestsCount++;
                    }
                    break;
            }
        }
    }
}
