package ru.yandex.webmaster3.monitoring.checklist;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.mutable.MutableLong;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemStorageType;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.checklist.dao.RealTimeSiteProblemsYDao;
import ru.yandex.webmaster3.storage.checklist.dao.SiteProblemsRecheckYDao;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.data.RealTimeSiteProblemInfo;
import ru.yandex.webmaster3.storage.checklist.data.SiteProblemRecheckInfo;

/**
 * @author avhaliullin
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ChecklistMonitoringService {

    private static final String SECTION_LABEL_VALUE = "checklist";
    private static final String LABEL_PROBLEM_NAME = "problem_type";
    private static final int AVERAGE_SENSORS_SIZE = 300;
    private static final Duration FROM = Duration.standardDays(13);// на всякий случай мониторим начинающие подтухать заявки
    private static final Duration TO = FROM.plus(Duration.standardDays(7)); // вот это ограничение просто чтобы слишком много не бегать. У нас был шанс отреагировать на проблему, пока шла эта неделя

    private final HandleCommonMetricsService handleCommonMetricsService;
    private final RealTimeSiteProblemsYDao realTimeSiteProblemsYDao;
    private final SiteProblemsRecheckYDao siteProblemsRecheckYDao;

    @Value("${webmaster3.monitoring.checklist.enabled}")
    private boolean enabled;
    @Value("${webmaster3.monitoring.checklist.refresh-interval}")
    private long refreshIntervalSeconds;

    @Scheduled(cron = "${webmaster3.monitoring.checklist.refresh-cron}")
    private void push() throws Exception {
        if (!enabled) {
            return;
        }
        Map<SiteProblemTypeEnum, MutableLong> problem2Complete = new EnumMap<>(SiteProblemTypeEnum.class);
        Map<SiteProblemTypeEnum, MutableLong> problem2Pending = new EnumMap<>(SiteProblemTypeEnum.class);
        Instant checkAfter = Instant.now().minus(TO);
        Instant checkBefore = Instant.now().minus(FROM);
        try (Batcher batcher = new Batcher(1000, batch -> processRecheckRequests(batch, problem2Complete, problem2Pending))) {
            siteProblemsRecheckYDao.foreach(request -> {
                if (!request.getProblemType().isDisabled()) {

                    if (request.isRecheckRequested() && request.getRequestDate().isAfter(checkAfter) && request.getRequestDate().isBefore(checkBefore)) {
                        batcher.accept(request);
                    }
                }
            });
        }
        List<SolomonSensor> sensors = new ArrayList<>();
        for (SiteProblemTypeEnum problemType : SiteProblemTypeEnum.ENABLED_PROBLEMS) {
            sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, problem2Complete.getOrDefault(problemType, new MutableLong(0L)).longValue())
                    .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, "rechecks-complete")
                    .withLabel(LABEL_PROBLEM_NAME, problemType.name())
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
            sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, problem2Pending.getOrDefault(problemType, new MutableLong(0L)).longValue())
                    .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, "rechecks-pending")
                    .withLabel(LABEL_PROBLEM_NAME, problemType.name())
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
        }
        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);
    }

    private void processRecheckRequests(List<SiteProblemRecheckInfo> requests, Map<SiteProblemTypeEnum, MutableLong> completeCount, Map<SiteProblemTypeEnum, MutableLong> pendingCount) {
        processRealTimeProblems(requests, completeCount, pendingCount);
    }

    private void processRealTimeProblems(List<SiteProblemRecheckInfo> requests, Map<SiteProblemTypeEnum, MutableLong> completeCount, Map<SiteProblemTypeEnum, MutableLong> pendingCount) {
        try {
            Set<WebmasterHostId> hosts = extractHosts(requests, SiteProblemStorageType.REAL_TIME);
            if (hosts.isEmpty()) {
                return;
            }
            Map<WebmasterHostId, List<RealTimeSiteProblemInfo>> hostProblems = realTimeSiteProblemsYDao.listSitesProblems(hosts);
            processProblemStates(requests, hostProblems, completeCount, pendingCount);
        } catch (WebmasterYdbException e) {
            throw new RuntimeException(e);
        }
    }

    private <T extends ProblemSignal> void processProblemStates(List<SiteProblemRecheckInfo> requests,
                                                                Map<WebmasterHostId, List<T>> state,
                                                                Map<SiteProblemTypeEnum, MutableLong> completeCount,
                                                                Map<SiteProblemTypeEnum, MutableLong> pendingCount) {
        requests.forEach(req -> {
            List<T> problems = state.getOrDefault(req.getHostId(), Collections.emptyList());
            Optional<T> problemStateOpt = problems.stream()
                    .filter(ps -> ps.getProblemType() == req.getProblemType())
                    .findAny();
            problemStateOpt.ifPresent(problemState -> {
                if (problemState.getLastUpdate() != null && problemState.getLastUpdate().isAfter(req.getRequestDate())) {
                    completeCount.computeIfAbsent(req.getProblemType(), ign -> new MutableLong(0L)).increment();
                } else {
                    pendingCount.computeIfAbsent(req.getProblemType(), ign -> new MutableLong(0L)).increment();
                }
            });
        });
    }

    private static Set<WebmasterHostId> extractHosts(List<SiteProblemRecheckInfo> requests, SiteProblemStorageType storageType) {
        return requests.stream()
                .filter(problem -> problem.getProblemType().getStorageType() == SiteProblemStorageType.REAL_TIME)
                .map(SiteProblemRecheckInfo::getHostId)
                .collect(Collectors.toSet());
    }

    private static class Batcher implements Consumer<SiteProblemRecheckInfo>, Closeable {
        private final int batchSize;
        private final List<SiteProblemRecheckInfo> buffer;
        private final Consumer<List<SiteProblemRecheckInfo>> consumer;

        public Batcher(int batchSize, Consumer<List<SiteProblemRecheckInfo>> consumer) {
            this.batchSize = batchSize;
            this.buffer = new ArrayList<>(batchSize);
            this.consumer = consumer;
        }

        private void flush() {
            consumer.accept(buffer);
            buffer.clear();
        }

        @Override
        public void close() {
            flush();
        }

        @Override
        public void accept(SiteProblemRecheckInfo siteProblemRecheckInfo) {
            buffer.add(siteProblemRecheckInfo);
            if (buffer.size() >= batchSize) {
                flush();
            }
        }
    }

}
