package ru.yandex.webmaster3.monitoring.clickhouse;

import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

import java.util.ArrayList;
import java.util.List;

/**
 * Мониторинг для активных процессов Clickhouse
 * Created by Oleg Bazdyrev on 18/07/2017.
 */
public class ClickhouseProcessesMonitoringService {

    private static final Logger log = LoggerFactory.getLogger(ClickhouseProcessesMonitoringService.class);
    private static final String PROCESSES_STATS_QUERY = "select count(), max(elapsed) from system.processes";
    private static final String ALL_PROCESSES_QUERY = "select * from system.processes";
    private static final String SECTION_LABEL_VALUE = "clickhouse";
    private static final String CLICKHOUSE_PROCESSES_TYPE = "clickhouse_processes";
    private static final String MAXIMUM_TYPE = "max_clickhouse_processes";
    private static final int AVERAGE_SENSORS_SIZE = 100;

    @Setter
    @Qualifier("legacyMdbClickhouseServer")
    private ClickhouseServer legacyMdbClickhouseServer;
    @Setter
    private HandleCommonMetricsService handleCommonMetricsService;
    @Setter
    private boolean enabled;
    @Setter
    private long refreshIntervalSeconds;

    @Scheduled(cron = "${webmaster3.monitoring.solomon.clickhouse.refresh-cron}")
    private void push() throws Exception {
        if (!enabled) {
            log.warn("Clickhouse processes monitoring service disabled");
            return;
        }

        List<SolomonSensor> sensors = new ArrayList<>();
        long maxQueue = 0L;
        double maxElapsed = 0.0;
        // собираем со всех серверов кликхауса инфу о текущих процессах
        for (ClickhouseHost clickhouseHost : legacyMdbClickhouseServer.getHosts()) {
            try {
                ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults().setHost(clickhouseHost);
                Pair<Long, Double> result = legacyMdbClickhouseServer.queryOne(context, PROCESSES_STATS_QUERY,
                        chRow -> Pair.of(chRow.getLongUnsafe("count()"), chRow.getDouble("max(elapsed)"))).get();
                maxQueue = Math.max(maxQueue, result.getLeft());
                maxElapsed = Math.max(maxElapsed, result.getRight());

                sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, result.getLeft())
                        .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                        .withLabel(SolomonSensor.LABEL_HOST, clickhouseHost.getHostURI().toString())
                        .withLabel(SolomonSensor.LABEL_DATA_TYPE, CLICKHOUSE_PROCESSES_TYPE)
                        .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

                sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, result.getRight())
                        .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                        .withLabel(SolomonSensor.LABEL_HOST, clickhouseHost.getHostURI().toString())
                        .withLabel(SolomonSensor.LABEL_DATA_TYPE, CLICKHOUSE_PROCESSES_TYPE)
                        .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

                // TODO временно, для разбора размера очереди в кликхаусе
                log.info("Logging out process list for host {}", clickhouseHost.getHostURI());
                legacyMdbClickhouseServer.queryAll(context, ALL_PROCESSES_QUERY, chRow -> {
                    // just log out query
                    log.debug("Query. Elapsed: {}, text: {}", chRow.getDouble("elapsed"), chRow.getString("query"));
                    return null;
                });

            } catch (ClickhouseException e) {
                log.error("Clickhouse host " + clickhouseHost + " returns error: ", e);
            }
        }
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, maxQueue)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, MAXIMUM_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, maxElapsed)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, MAXIMUM_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE * legacyMdbClickhouseServer.getHosts().size() * 2);
    }
}
