package ru.yandex.webmaster3.monitoring.queue.clickhouse;

import java.util.ArrayList;
import java.util.List;

import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.annotation.Scheduled;

import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.storage.clickhouse.replication.dao.ClickhouseReplicationQueueYDao;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationTaskGroup;
import ru.yandex.webmaster3.storage.ytimport.dao.YtClickhouseImportQueueYDao;
import ru.yandex.webmaster3.storage.ytimport.dao.YtClickhouseImportQueueYDao.YtClickhouseImportRecord;

/**
 * Мониторинг очередей репликации и импорта в кликхаус
 * Created by Oleg Bazdyrev on 18/07/2017.
 */
public class ClickhouseQueuesMonitoringService {

    private static final Logger log = LoggerFactory.getLogger(ClickhouseQueuesMonitoringService.class);
    private static final String SECTION_LABEL_VALUE = "clickhouse";
    private static final String REPLICATION_TYPE = "replication";
    private static final String IMPORT_TYPE = "import";
    private static final int AVERAGE_SENSORS_SIZE = 100;

    @Setter
    private HandleCommonMetricsService handleCommonMetricsService;
    @Setter
    private ClickhouseReplicationQueueYDao clickhouseReplicationQueueYDao;
    @Setter
    private YtClickhouseImportQueueYDao ytClickhouseImportQueueYDao;
    @Setter
    private boolean enabled;
    @Setter
    private long refreshIntervalSeconds;

    @Scheduled(cron = "${webmaster3.monitoring.solomon.clickhouse-queues.refresh-cron}")
    private void push() throws Exception {
        if (!enabled) {
            log.warn("Clickhouse queues monitoring service disabled");
            return;
        }

        long now = System.currentTimeMillis();
        long replicationQueueSize = 0;
        long replicationQueueAge = now;
        // получаем возраст и размер очереди
        for (ClickhouseReplicationTaskGroup task : clickhouseReplicationQueueYDao.listUnfinishedTasks()) {
            replicationQueueSize++;
            replicationQueueAge = Math.min(replicationQueueAge, task.getCommand().getCreateDate().getMillis());
        }

        long importQueueSize = 0;
        long importQueueAge = now;
        // получаем возраст и размер очереди
        for (YtClickhouseImportRecord task : ytClickhouseImportQueueYDao.listAll()) {
            if (!task.getState().isTerminal()) {
                importQueueSize++;
                importQueueAge = Math.min(importQueueAge, task.getId().timestamp());
            }
        }

        List<SolomonSensor> sensors = new ArrayList<>();
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, replicationQueueSize)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, REPLICATION_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, Math.max(now - replicationQueueAge, 0) / 1000)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, REPLICATION_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, importQueueSize)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IMPORT_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, Math.max(now - importQueueAge, 0) / 1000)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IMPORT_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE * sensors.size());
    }

}
