package ru.yandex.webmaster3.monitoring.queue.delurl;

import lombok.RequiredArgsConstructor;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.delurl.DelUrlRequest;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.*;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.delurl.DelUrlRequestsService;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataState;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataType;
import ru.yandex.webmaster3.storage.monitoring.common.dao.MonitoringDataStateYDao;
import ru.yandex.webmaster3.storage.spam.DeepSpamHostFilter;
import ru.yandex.webmaster3.storage.spam.FastSpamHostFilter;

import java.util.*;

import static org.joda.time.DateTimeConstants.SECONDS_PER_MINUTE;

/**
 * @author leonidrom
 */
@Slf4j
@Service
public class DelUrlMonitoringService {
    private static final String SECTION_LABEL_VALUE = "delurl";
    private static final String NEW_REQUESTS_DATA_TYPE = "new_requests";
    private static final String IN_PROGRESS_REQUESTS_DATA_TYPE = "in_progress_requests";
    private static final String PROCESSED_REQUESTS_DATA_TYPE = "processed_requests";
    private static final String ERROR_REQUESTS_DATA_TYPE = "error_requests";

    private static final int AVERAGE_SENSORS_SIZE = 600;

    private static final List<Duration> BUCKETS = Arrays.asList(
            Duration.standardMinutes(10),
            Duration.standardMinutes(30),
            Duration.standardHours(1),
            Duration.standardHours(2),
            Duration.standardHours(4),
            Duration.standardHours(6),
            Duration.standardHours(12),
            Duration.standardDays(1)
    );

    private final HandleCommonMetricsService handleCommonMetricsService;
    private final DeepSpamHostFilter deepSpamHostFilter;
    private final FastSpamHostFilter fastSpamHostFilter;
    private final MonitoringDataStateYDao monitoringDataStateYDao;
    private final DelUrlRequestsService delurlRequestsService;
    private final AbtService abtService;

    private long refreshIntervalSeconds = 3600;

    @Autowired
    public DelUrlMonitoringService(
            HandleCommonMetricsService handleCommonMetricsService,
            DeepSpamHostFilter deepSpamHostFilter,
            FastSpamHostFilter fastSpamHostFilter,
            MonitoringDataStateYDao monitoringDataStateYDao,
            DelUrlRequestsService delurlRequestsService, AbtService abtService) {
        this.handleCommonMetricsService = handleCommonMetricsService;
        this.deepSpamHostFilter = deepSpamHostFilter;
        this.fastSpamHostFilter = fastSpamHostFilter;
        this.monitoringDataStateYDao = monitoringDataStateYDao;
        this.delurlRequestsService = delurlRequestsService;
        this.abtService = abtService;
    }

    // важно чтобы эта таска не стартовала одновременно с UpdateUrlStatePeriodicTask
    @Scheduled(cron = "${webmaster3.monitoring.solomon.addurl.refresh-cron}")
    private void push() throws Exception {
        log.info("Started gathering stats.");
        long now = System.currentTimeMillis();
        var lastRunDS = monitoringDataStateYDao.getValue(MonitoringDataType.LAST_ADDURL_MONITORING_RUN);
        long lastRunMillis = lastRunDS == null? now - refreshIntervalSeconds * 1000 : lastRunDS.getLastUpdate().getMillis();

        Map<SolomonKey, SolomonCounterImpl> sensorsAcc = new HashMap<>();
        SolomonHistogram<Duration> inProgressAgeHist = createHistogram(sensorsAcc);
        DelUrlRequestsStatistics stats = new DelUrlRequestsStatistics(lastRunMillis);

        // собираем статистику
        List<DelUrlRequest> reqs = delurlRequestsService.list(
                DateTime.now().minus(DelUrlRequest.STALE_REQUEST_AGE).minusDays(1), // вычитываем с запасом, чтобы увидеть STALE запросы
                DateTime.now().minus(Duration.standardMinutes(5)) // не хотим вычитывать живые данные, чтобы не было конфликта транзакций
        );

        reqs.forEach(req -> {
            WebmasterHostId hostId = req.getHostId();
            boolean spam = fastSpamHostFilter.checkHost(hostId) ||
                    deepSpamHostFilter.checkHost(hostId) ||
                    !abtService.isInExperiment(hostId, "DELURL_SAMOVAR");
            if (!spam) {
                stats.accept(now, req, inProgressAgeHist);
            }
        });

        // отправляем сенсоры
        List<SolomonSensor> sensors = new ArrayList<>();
        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, (now - stats.minNewRequestDate) / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, NEW_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.newRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, NEW_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, (now - stats.minInProgressRequestDate) / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.inProgressRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.processedRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, PROCESSED_REQUESTS_DATA_TYPE));

        sensors.add(SolomonSensor.createAligned(now, SECONDS_PER_MINUTE, stats.errorRequestsCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, ERROR_REQUESTS_DATA_TYPE));

        sensorsAcc.forEach((key, counter) -> {
            sensors.add(SolomonSensor.createAligned(key, now, SECONDS_PER_MINUTE, counter.getAsLong()));
        });

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);
        monitoringDataStateYDao.update(new MonitoringDataState(MonitoringDataType.LAST_ADDURL_MONITORING_RUN, "", DateTime.now()));

        log.info("Finished gathering stats: {}", stats);
    }


    private static SolomonHistogram<Duration> createHistogram(Map<SolomonKey, SolomonCounterImpl> sensorsAcc) {
        return SolomonHistogramImpl.create(duration -> createCounter(sensorsAcc, duration), BUCKETS);
    }

    private static SolomonCounter createCounter(Map<SolomonKey, SolomonCounterImpl> sensorsAcc, Duration duration) {
        return sensorsAcc.computeIfAbsent(SolomonKey.create(
                SolomonKey.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                        .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                        .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE)
                        .withLabel(SolomonKey.LABEL_TIME_BUCKET, "<" + duration.getStandardMinutes() + "m"),
                ign -> SolomonCounterImpl.create(false)
        );
    }

    @RequiredArgsConstructor
    @ToString
    private static class DelUrlRequestsStatistics {
        final long lastRunMillis;

        long minNewRequestDate = System.currentTimeMillis();
        long newRequestsCount = 0;
        long minInProgressRequestDate = System.currentTimeMillis();
        long inProgressRequestsCount = 0;
        long processedRequestsCount = 0;
        long errorRequestsCount = 0;
        DelUrlRequest minNewRequest;

        void accept(long now, DelUrlRequest req, SolomonHistogram<Duration> inProgressAgeHist) {
            long updateTS = req.getUpdateDate().getMillis();
            switch (req.getState()) {
                case NEW:
                    long addTS = req.getAddDate().getMillis();
                    minNewRequestDate = Math.min(addTS, minNewRequestDate);
                    if (minNewRequestDate == addTS) {
                        minNewRequest = req;
                    }
                    newRequestsCount++;
                    break;

                case IN_PROGRESS:
                    minInProgressRequestDate = Math.min(updateTS, minInProgressRequestDate);
                    inProgressRequestsCount++;
                    var requestAge = Duration.millis(now - req.getAddDate().getMillis());
                    inProgressAgeHist.update(requestAge);
                    break;

                case ACCEPTED:
                case REJECTED:
                    if (updateTS >= lastRunMillis) {
                        processedRequestsCount++;
                    }
                    break;

                case ERROR:
                    if (updateTS >= lastRunMillis) {
                        errorRequestsCount++;
                    }
                    break;
            }
        }
    }
}
