package ru.yandex.webmaster3.monitoring.queue.mobile;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.mutable.MutableLong;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.*;
import ru.yandex.webmaster3.storage.mobile.dao.HostMobileAuditQueueYDao;

import java.util.*;


/**
 * @author leonidrom
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class MobileAuditMonitoringService {
    private static final int AVERAGE_SENSORS_SIZE = 500;
    private static final String SECTION_LABEL_VALUE = "host_mobile_audit";
    private static final String IN_PROGRESS_REQUESTS_DATA_TYPE = "in_progress_requests";
    private static final String CASSANDRA_QUEUE_TYPE = "cassandra_queue";
    private static final int REFRESH_INTERVAL_SECONDS = 1800;

    private final HostMobileAuditQueueYDao hostMobileAuditQueueYDao;
    private final HandleCommonMetricsService handleCommonMetricsService;

    private static final List<Duration> BUCKETS = Arrays.asList(
            Duration.standardMinutes(10),
            Duration.standardMinutes(30),
            Duration.standardHours(1),
            Duration.standardHours(2),
            Duration.standardHours(4),
            Duration.standardHours(6),
            Duration.standardHours(12),
            Duration.standardDays(1),
            Duration.standardDays(3),
            Duration.standardDays(7),
            Duration.standardDays(14)
    );

    @Scheduled(cron = "0 0/30 * * * *")
    private void push() {
        log.info("Started collecting metrics for host mobile audit queue");

        List<SolomonSensor> sensors = new ArrayList<>();
        Map<SolomonKey, SolomonCounterImpl> inProgressHistogramSensors = new HashMap<>();
        SolomonHistogram<Duration> inProgressHistogram = createHistogram(inProgressHistogramSensors);
        Pair<Integer, Long> unprocessedCassandraRequestsStats = getUnprocessedCassandraRequestsStats(inProgressHistogram);

        long now = System.currentTimeMillis();
        sensors.add(SolomonSensor.createAligned(now, REFRESH_INTERVAL_SECONDS, unprocessedCassandraRequestsStats.getLeft())
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, CASSANDRA_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
        sensors.add(SolomonSensor.createAligned(now, REFRESH_INTERVAL_SECONDS, unprocessedCassandraRequestsStats.getRight() / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, CASSANDRA_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

        inProgressHistogramSensors.forEach((key, counter) -> {
            sensors.add(SolomonSensor.createAligned(key, now, REFRESH_INTERVAL_SECONDS, counter.getAsLong()));
        });

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);

        log.info("Done collecting metrics for host mobile audit queue");
    }

    private Pair<Integer, Long> getUnprocessedCassandraRequestsStats(SolomonHistogram<Duration> inProgress) {
        long now = System.currentTimeMillis();
        MutableInt count = new MutableInt(0);
        MutableLong minTime = new MutableLong(Long.MAX_VALUE);

        hostMobileAuditQueueYDao.forEachRequest(req -> {
            long addDate = req.getAddDate().getMillis();
            var requestAge = Duration.millis(now - addDate);

            inProgress.update(requestAge);
            minTime.setValue(Math.min(minTime.getValue(), addDate));
            count.increment();
        });

        if (minTime.getValue() == Long.MAX_VALUE) {
            minTime.setValue(now);
        }

        return Pair.of(count.getValue(), now - minTime.getValue());
    }

    private static SolomonHistogram<Duration> createHistogram(Map<SolomonKey, SolomonCounterImpl> sensorsAcc) {
        return SolomonHistogramImpl.create(duration -> createCounter(sensorsAcc, duration), BUCKETS);
    }

    private static SolomonCounter createCounter(Map<SolomonKey, SolomonCounterImpl> sensorsAcc, Duration duration) {
        return sensorsAcc.computeIfAbsent(SolomonKey.create(
                SolomonKey.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                        .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                        .withLabel(SolomonSensor.LABEL_DATA_TYPE, IN_PROGRESS_REQUESTS_DATA_TYPE)
                        .withLabel(SolomonKey.LABEL_TIME_BUCKET, "<" + duration.getStandardMinutes() + "m"),
                ign -> SolomonCounterImpl.create(false)
        );
    }
}
