package ru.yandex.webmaster3.monitoring.queue.moderation;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.base.Stopwatch;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.mutable.MutableLong;
import org.apache.commons.lang3.mutable.MutableObject;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.monitoring.common.YtQueueUtils;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequest;
import ru.yandex.webmaster3.storage.host.moderation.regions.dao.HostRegionsModerationRequestsYDao;
import ru.yandex.webmaster3.storage.host.moderation.regions.dao.HostRegionsModerationYtRequestsYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import static ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequestStatus.IN_MODERATION;

/**
 * @author leonidrom
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class HostRegionsModerationMonitoringService {
    private static final Logger log = LoggerFactory.getLogger(HostRegionsModerationMonitoringService.class);

    private static final Predicate<String> RESULTS_TABLES_FILTER = t -> t.startsWith("req") || t.startsWith("auto");
    private static final String SECTION_LABEL_VALUE = "host_regions";
    private static final String CASSANDRA_QUEUE_TYPE = "cassandra_queue";
    private static final String YT_QUEUE_TYPE = "yt_queue";
    private static final String HOST_REGIONS_QUEUE_TYPE = "host_regions_queue";
    private static final String YT_RESULTS_QUEUE_TYPE = "yt_results";
    private static final int AVERAGE_SENSORS_SIZE = 500;
    private static DateTime SKIP_REQUESTS_BEFORE_DATE = DateTime.parse("2018-01-25T19:54:52.128+03:00");

    private final HostRegionsModerationYtRequestsYDao hrmYtRequestsYDao;
    private final HostRegionsModerationRequestsYDao hrmRequestsYDao;
    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;

    private final HandleCommonMetricsService handleCommonMetricsService;
    private final YtService ytService;

    @Value("${webmaster3.monitoring.solomon.hostregions.enabled}")
    private boolean enabled;
    @Value("${webmaster3.monitoring.solomon.hostregions.refresh-interval}")
    private long refreshIntervalSeconds;
    @Value("${external.yt.service.hahn.root.default}/export/yang/host_regions")
    private YtPath ytRequestsPath;
    @Value("${external.yt.service.hahn.root.default}/import/yang/host_regions")
    private YtPath ytResultsPath;

    @Scheduled(cron = "${webmaster3.monitoring.solomon.hostregions.refresh-cron}")
    private void push() throws Exception {
        if (!enabled) {
            log.warn("Host regions moderation metrics service disabled");
            return;
        }
        log.info("Started collecting metrics for host regions moderation");
        List<SolomonSensor> sensors = new ArrayList<>();

        Stopwatch sw = Stopwatch.createStarted();
        Pair<Integer, Long> unprocessedYdbRequestsStats = getUnprocessedYdbRequestsStats();
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYdbRequestsStats.getLeft())
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, CASSANDRA_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYdbRequestsStats.getRight() / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, CASSANDRA_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));
        log.info("Collected stats for cassandra requests in {}s", sw.stop().elapsed(TimeUnit.SECONDS));

        sw.reset().start();
        YtQueueUtils.QueueStats unprocessedYtRequestsStats = YtQueueUtils.getQueueStats(
                ytService, ytRequestsPath, YtQueueUtils.ALL_TABLES_FILTER, YtQueueUtils.JAVA_TIMESTAMP_EXTRACTOR);
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYtRequestsStats.tablesCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, YT_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYtRequestsStats.maxAgeInMillis / 1000)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, YT_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));
        log.info("Collected stats for unprocessed YT requests in {}s", sw.stop().elapsed(TimeUnit.SECONDS));

        sw.reset().start();
        YtQueueUtils.QueueStats unprocessedYtResultsStats = YtQueueUtils.getQueueStats(
                ytService, ytResultsPath, RESULTS_TABLES_FILTER, YtQueueUtils.JAVA_TIMESTAMP_EXTRACTOR);
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYtResultsStats.tablesCount)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, YT_RESULTS_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedYtResultsStats.maxAgeInMillis / 1000)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, YT_RESULTS_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));
        log.info("Collected stats for unprocessed YT results in {}s", sw.stop().elapsed(TimeUnit.SECONDS));

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);
        log.info("Done collecting metrics for host regions moderation");
    }

    @Scheduled(cron = "${webmaster3.monitoring.solomon.hostregions.refresh-heavy-cron}")
    private void pushHeavyMetrics() {
        if (!enabled) {
            log.warn("Host regions moderation metrics service disabled");
            return;
        }
        log.info("Started collecting heavy metrics for host regions moderation");

        Stopwatch sw = Stopwatch.createStarted();
        NavigableMap<Long, HostRegionsModerationRequest> unprocessedRequestsMap = getUnprocessedRequestsStats();
        log.info("Collected stats for whole host regions moderation process in {}s", sw.stop().elapsed(TimeUnit.SECONDS));

        List<SolomonSensor> sensors = new ArrayList<>();
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, unprocessedRequestsMap.size())
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, HOST_REGIONS_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

        Map.Entry<Long, HostRegionsModerationRequest> oldestEntry = unprocessedRequestsMap.firstEntry();
        if (oldestEntry == null) {
            return;
        }

        log.info("Oldest unprocessed request: {}", oldestEntry.getValue());
        sensors.add(SolomonSensor.createAligned(refreshIntervalSeconds, oldestEntry.getKey() / 1000L)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, HOST_REGIONS_QUEUE_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE));

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);

        unprocessedRequestsMap.entrySet().stream().limit(150).forEach(e -> {
            log.info("Unprocessed request: {}", e.getValue());
        });

        log.info("Done collecting heavy metrics for host regions moderation");
    }

    /**
     * Возвращает пару (размер очереди, возраст) для очереди запросов в Кассандре,
     * ожидающих отправку в Yt.
     */
    private Pair<Integer, Long> getUnprocessedYdbRequestsStats() {
        try {
            long now = System.currentTimeMillis();
            MutableInt count = new MutableInt(0);
            MutableLong minTime = new MutableLong(Long.MAX_VALUE);
            hrmYtRequestsYDao.forEach(
                    req -> {
                        var reqId = req.getRequestId();
                        long reqTime = UUIDs.unixTimestamp(reqId);
                        if (minTime.getValue() > reqTime) {
                            minTime.setValue(reqTime);
                        }
                        count.increment();
                    }
            );
            if (minTime.getValue() == Long.MAX_VALUE) {
                minTime.setValue(now);
            }
            return Pair.of(count.getValue(), now - minTime.getValue());
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to obtain list of requests to process",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    /**
     * Возвращает мапу (возраст, заявка) дл всех необработанных заявок.
     */
    private NavigableMap<Long, HostRegionsModerationRequest> getUnprocessedRequestsStats() {
        long now = System.currentTimeMillis();
        NavigableMap<Long, HostRegionsModerationRequest> unprocessedRequestsMap = new TreeMap<>(Comparator.reverseOrder());

        MutableObject<HostRegionsModerationRequest> maxReqObj = new MutableObject<>(null);
        MutableObject<WebmasterHostId> curHostIdObj = new MutableObject<>(null);
        try {
            hrmRequestsYDao.forEach(req -> {
                var hostId = req.getHostId();
                if (!allVerifiedHostsCacheService.contains(hostId) || SKIP_REQUESTS_BEFORE_DATE.isAfter(req.getUpdateDate())) {
                    return;
                }

                var curHostId = curHostIdObj.getValue();
                var maxReq = maxReqObj.getValue();
                if (!hostId.equals(curHostId)) {
                    if (maxReq != null && maxReq.getStatus() == IN_MODERATION) {
                        unprocessedRequestsMap.put(now - maxReq.getUpdateDate().getMillis(), maxReq);
                    }

                    maxReqObj.setValue(req);
                    curHostIdObj.setValue(hostId);
                } else {
                    if (req.getUpdateDate().isAfter(maxReq.getUpdateDate())) {
                        maxReqObj.setValue(req);
                    }
                }
            });

            var maxReq = maxReqObj.getValue();
            if (maxReq != null && maxReq.getStatus() == IN_MODERATION) {
                unprocessedRequestsMap.put(now - maxReq.getUpdateDate().getMillis(), maxReq);
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to obtain list of requests to process",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }

        return unprocessedRequestsMap;
    }
}
