package ru.yandex.webmaster3.monitoring.queue.sitemap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.sitemap.HostSitemap;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.storage.host.dao.HostsYDao;
import ru.yandex.webmaster3.storage.sitemap.SitemapRecrawlRequestService;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapRecrawlRequest;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapRecrawlRequestsYDao;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapsCHDao;
import ru.yandex.webmaster3.core.sitemap.SitemapInfo;

/**
 * @author leonidrom
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SitemapRecrawlMonitoringService {
    public static final String SECTION_LABEL_VALUE = "sitemap_recrawl";
    public static final String DATA_TYPE_LABEL_VALUE = "pending_requests";

    private static final int AVERAGE_SENSORS_SIZE = 600;
    private static final int TOTAL_THREADS = 8;

    private final HandleCommonMetricsService handleCommonMetricsService;
    private final HostsYDao hostsYDao;
    private final SitemapRecrawlRequestsYDao sitemapRecrawlRequestsYDao;
    private final SitemapRecrawlRequestService sitemapRecrawlRequestService;
    private final SitemapsCHDao sitemapsCHDao;
    private long refreshIntervalSeconds = Duration.standardHours(3).getStandardSeconds();

    @Scheduled(cron = "0 0 0/3 * * *")
    private void push() throws Exception {
        log.info("Started collecting metrics for sitemaps recrawl");
        var stats = collectStats();
        log.info("Finished collecting metrics for sitemaps recrawl");

        long now = System.currentTimeMillis();
        List<SolomonSensor> sensors = new ArrayList<>();

        SolomonKey baseKeyDataAge = SolomonKey.create()
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE);

        stats.pendingRequestsAge.forEach((period, count) -> {
            var sensor = SolomonSensor.createAligned(now, refreshIntervalSeconds, count)
                    .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                    .withLabel("sitemap_recrawl_time_bucket", period.durationWithUnit())
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, DATA_TYPE_LABEL_VALUE);

            sensors.add(sensor);
        });

        var sensor = SolomonSensor.createAligned(now, refreshIntervalSeconds, stats.pendingRequests)
                .withLabel(SolomonSensor.LABEL_SECTION, SECTION_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.COUNT)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, DATA_TYPE_LABEL_VALUE);
        sensors.add(sensor);

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);

        log.info("Stats: {}", stats);
    }

    @NotNull
    private Stats collectStats() throws Exception {
        Set<WebmasterHostId> hosts = new HashSet<>();
        sitemapRecrawlRequestsYDao.forEachDistinctHost(hosts::add);
        log.info("Got {} hosts", hosts.size());

        ExecutorService executorService = ru.yandex.common.util.concurrent.Executors.newBlockingFixedThreadPool(
                TOTAL_THREADS, TOTAL_THREADS,
                0, TimeUnit.MILLISECONDS,
                new ArrayBlockingQueue<>(TOTAL_THREADS),
                Executors.defaultThreadFactory());

        var stats = new Stats();
        try {
            List<Future<Stats>> futures = hosts.stream()
                    .map(hostId -> executorService.submit(() -> processRequests(hostId)))
                    .collect(Collectors.toList());

            for (Future<Stats> f : futures) {
                stats.accumulate(f.get(2, TimeUnit.HOURS));
            }
        } finally {
            executorService.shutdownNow();
        }

        return stats;
    }

    @NotNull
    private Stats processRequests(WebmasterHostId hostId) {
        var stats = new Stats();

        if (!hostsYDao.isHostAdded(hostId)) {
            return stats;
        }

        var requests = sitemapRecrawlRequestService.getLatestHostRequests(hostId);
        requests.forEach(req -> {
            var reqStats = processRequest(hostId, req);
            stats.accumulate(reqStats);
        });

        return stats;
    }

    @NotNull
    private Stats processRequest(WebmasterHostId hostId, SitemapRecrawlRequest req) {
        HostSitemap sitemap = sitemapsCHDao.getSitemap(hostId, req.getParentSitemapId(), req.getSitemapId()).map(SitemapInfo::toHostSitemap).orElse(null);
        if (sitemap == null) {
            return Stats.newSkipped();
        }

        DateTime lastAccessDate = sitemap.getInfo() == null ? null : sitemap.getInfo().getLastAccessDate();
        DateTime requestDate = req.getRequestDate();
        boolean isPending = true;
        if (lastAccessDate != null) {
            isPending = !lastAccessDate.isAfter(requestDate);
        }

        var stats = new Stats();
        if (isPending) {
            stats = Stats.newPending(requestDate);
            Period p = stats.pendingRequestsAge.keySet().stream().findFirst().get();
            if (p.getDuration().isLongerThan(Period.P_6D.getDuration())) {
                log.info("Stale sitemap: {}, {}, {}", p, hostId, req);
            }
        } else {
            stats = Stats.newProcessed();
        }

        return stats;
    }

    private enum Period {
        P_1H(1),
        P_6H(6),
        P_12H(12),
        P_1D(24),
        P_2D(24 * 2),
        P_3D(24 * 3),
        P_4D(24 * 4),
        P_5D(24 * 5),
        P_6D(24 * 6),
        P_7D(24 * 7),
        P_14D(24 * 14),
        P_21D(24 * 21),
        P_30D(24 * 30),
        P_MORE(Integer.MAX_VALUE);

        private final Duration hours;

        Period(long hours) {
            this.hours = Duration.standardHours(hours);
        }

        public Duration getDuration() {
            return hours;
        }

        static Period get(long hours) {
            for (Period p : Period.values()) {
                if (hours < p.getDuration().getStandardHours()) {
                    return p;
                }
            }

            return Period.P_MORE;
        }

        String durationWithUnit() {
            if (this == P_MORE) {
                return "<*";
            } else {
                return "<" + hours.getStandardHours() + "h";
            }
        }
    }

    @ToString
    public static class Stats {
        private long pendingRequests;
        private long processedRequests;
        private long skippedRequests;
        private Map<Period, Long> pendingRequestsAge;

        Stats() {
            pendingRequestsAge = new HashMap<>();
            populateBucketMap(pendingRequestsAge);
        }

        void accumulate (Stats s) {
            pendingRequests += s.pendingRequests;
            skippedRequests += s.skippedRequests;
            processedRequests += s.processedRequests;
            s.pendingRequestsAge.forEach((k, v) -> pendingRequestsAge.merge(k, v, Long::sum));
        }

        static Stats newSkipped() {
            var stats = new Stats();
            stats.skippedRequests = 1;
            return stats;
        }

        static Stats newProcessed() {
            var stats = new Stats();
            stats.processedRequests = 1;
            return stats;
        }

        static Stats newPending(DateTime requestDate) {
            Stats stats = new Stats();
            stats.pendingRequests = 1;
            Duration pendingDuration = new Duration(requestDate, DateTime.now());
            stats.pendingRequestsAge.put(Period.get(pendingDuration.getStandardHours()), 1L);

            return stats;
        }

        private void populateBucketMap(Map<Period, Long> bucketMap) {
            for (Period p : Period.values()) {
                bucketMap.put(p, 0L);
            }
        }
    }
}
