package ru.yandex.webmaster3.monitoring.solomon.dao;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import org.joda.time.Duration;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.monitoring.solomon.trigger.data.WebmasterTrigger;
import ru.yandex.webmaster3.monitoring.solomon.trigger.data.WebmasterTriggerInfo;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Delete;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Select;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Upsert;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;


/**
 * @author kravchenko99
 * @date 11/9/20
 */

@Repository("webmasterTriggersYDao")
public class WebmasterTriggersYDao extends AbstractYDao {
    private static final String TABLE_NAME = "webmaster_triggers";

    public WebmasterTriggersYDao() {
        super(PREFIX_MONITORING, TABLE_NAME);
    }

    public List<WebmasterTriggerInfo> listAllTriggers() {
        final Select st = select(MAPPER);
        st.order(F.TRIGGER_ID.asc());

        return queryForList(st, MAPPER);
    }

    public void updateTrigger(WebmasterTriggerInfo triggerInfo) {
        final Upsert upsert = upsert(
                F.TRIGGER_ID.value(triggerInfo.getId()),
                F.NAME.value(triggerInfo.getName()),
                F.COMMON_LABELS.value(triggerInfo.getCommonLabels().getLabels()),
                F.PROJECT.value(triggerInfo.getProject()),
                F.SERVICE.value(triggerInfo.getService()),
                F.DASHBOARD_CATEGORY.value(triggerInfo.getDashboardCategory()),
                F.DATA_INTERVAL.value(triggerInfo.getDataDuration()),
                F.TRIGGER.value(triggerInfo.getTrigger())
        );
        execute(upsert);
    }

    public WebmasterTriggerInfo getTriggerInfo(String id) {
        final Select.Where st = select(MAPPER)
                .where(F.TRIGGER_ID.eq(id));
        return queryOne(st.getStatement(), MAPPER);
    }

    public void deleteTrigger(String id) {
        final Delete st = delete();
        st.where(F.TRIGGER_ID.eq(id));
        execute(st);
    }

    private final static DataMapper<WebmasterTriggerInfo> MAPPER = DataMapper.create(
            F.TRIGGER_ID, F.NAME, F.COMMON_LABELS, F.PROJECT, F.SERVICE, F.HOST, F.DASHBOARD_CATEGORY, F.DATA_INTERVAL,
            F.TRIGGER, WebmasterTriggerInfo::new
    );

    private interface F {
        TypeReference<Map<String, String>> MAP = new TypeReference<>() {
        };

        Field<String> TRIGGER_ID = Fields.stringField("trigger_id");
        Field<String> NAME = Fields.stringField("name");
        Field<Map<String, String>> COMMON_LABELS = Fields.mapField("common_labels", MAP);
        Field<String> PROJECT = Fields.stringField("project");
        Field<String> SERVICE = Fields.stringField("service");
        Field<String> HOST = Fields.stringField("host").makeOptional();
        Field<String> DASHBOARD_CATEGORY = Fields.stringField("dashboard_category");
        Field<Duration> DATA_INTERVAL = Fields.jodaDurationField("data_interval").makeOptional();
        Field<WebmasterTrigger> TRIGGER = Fields.jsonField2("trigger", WebmasterTrigger.class);

    }
}
