package ru.yandex.webmaster3.monitoring.solomon.trigger.actions;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.solomon_api.model.SolomonAlertEvaluationStatusDto;
import ru.yandex.solomon_api.model.SolomonSubAlertListItemDto;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.openapi.internal.ApiException;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.monitoring.solomon.api.SolomonControlApiV2Service;
import ru.yandex.webmaster3.monitoring.solomon.trigger.actions.GetPeriodicTasksAlertsStateAction.Request;
import ru.yandex.webmaster3.monitoring.solomon.trigger.actions.GetPeriodicTasksAlertsStateAction.Response;

/**
 * @author leonidrom
 */
@ReadAction
public class GetPeriodicTasksAlertsStateAction extends Action<Request, Response> {
    private static final Logger log = LoggerFactory.getLogger(GetPeriodicTasksAlertsStateAction.class);
    private SolomonControlApiV2Service solomonControlApiV2Service;
    private final static Set<String> PERIODIC_TASK_TYPES = Arrays.stream(PeriodicTaskType.values()).map(Enum::name).collect(Collectors.toSet());

    @Override
    public Response process(Request request) throws WebmasterException {
        // periodic task failures sub alerts in red (ALARM) state
        Map<String, SolomonAlertEvaluationStatusDto> statuses = new HashMap<>();
        try {

            List<SolomonSubAlertListItemDto> subAlertsRed = solomonControlApiV2Service.listSubAlerts("wmc_periodic_task_failures", Collections.singletonList("ALARM"),
                    Arrays.asList("trafficLight.color", "trafficLight.value", "name"))
                    .stream().filter(x -> PERIODIC_TASK_TYPES.contains(x.getAnnotations().getOrDefault("name", ""))).collect(Collectors.toList());
            for (SolomonSubAlertListItemDto subAlert : subAlertsRed) {
                SolomonAlertEvaluationStatusDto.CodeEnum code = SolomonAlertEvaluationStatusDto.CodeEnum.fromValue(subAlert.getEvaluationStatusCode().getValue());
                SolomonAlertEvaluationStatusDto status =
                        new SolomonAlertEvaluationStatusDto().code(code).annotations(subAlert.getAnnotations());

                statuses.put("wmc_periodic_task_failures/subAlerts/" + subAlert.getId(), status);
            }
        } catch (ApiException e) {
            log.error("Failed to get alert status", e);
            // ignore
        }

        return new Response(statuses);
    }

    public static class Request implements ActionRequest {
    }

    public static class Response implements ActionResponse.NormalResponse {
        private final Map<String, SolomonAlertEvaluationStatusDto> statuses;

        public Response(Map<String, SolomonAlertEvaluationStatusDto> statuses) {
            this.statuses = statuses;
        }

        public Map<String, SolomonAlertEvaluationStatusDto> getStatuses() {
            return statuses;
        }
    }

    @Required
    public void setSolomonControlApiV2Service(SolomonControlApiV2Service solomonControlApiV2Service) {
        this.solomonControlApiV2Service = solomonControlApiV2Service;
    }
}
