package ru.yandex.webmaster3.monitoring.urlcheck;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.SolomonGroupingUtil;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataState;
import ru.yandex.webmaster3.storage.monitoring.common.MonitoringDataType;
import ru.yandex.webmaster3.storage.monitoring.common.dao.MonitoringDataStateYDao;
import ru.yandex.webmaster3.storage.url.checker3.service.UrlCheckDataBlocksService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.joda.time.DateTimeConstants.SECONDS_PER_MINUTE;

/**
 * @author leonidrom
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class UrlCheckMonitoringService {
    private static final int AVERAGE_SENSORS_SIZE = 600;
    private static final long refreshIntervalSeconds = 300;

    private final MonitoringDataStateYDao monitoringDataStateYDao;
    private final UrlCheckDataBlocksService urlCheckDataBlocksService;
    private final HandleCommonMetricsService handleCommonMetricsService;

    @Scheduled(cron = "0 0/5 * * * *")
    private void push() throws Exception {
        log.info("Started requests stats.");

        var now = DateTime.now();
        var lastRunDS = monitoringDataStateYDao.getValue(MonitoringDataType.LAST_URLCHECK_MONITORING_RUN);
        long lastRunMillis = lastRunDS == null? now.getMillis() - refreshIntervalSeconds * 1000 : lastRunDS.getLastUpdate().getMillis();
        var lastRunDate = new DateTime(lastRunMillis);

        var blockReqs = urlCheckDataBlocksService.getWithoutData(lastRunDate, now);
        log.info("Finished gathering requests stats.");

        Map<Pair<String, String>, Long> countMap = new HashMap<>();
        blockReqs.forEach(r -> {
            countMap.merge(Pair.of(r.getBlockType().name(), r.getFetchState().name()), 1L, Long::sum);
            countMap.merge(Pair.of(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE, r.getFetchState().name()), 1L, Long::sum);
            countMap.merge(Pair.of(r.getBlockType().name(), SolomonGroupingUtil.AGGREGATE_LABEL_VALUE), 1L, Long::sum);
            countMap.merge(Pair.of(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE, SolomonGroupingUtil.AGGREGATE_LABEL_VALUE), 1L, Long::sum);

        });

        // отправляем сенсоры
        List<SolomonSensor> sensors = new ArrayList<>();
        countMap.forEach((typeStatePair, cnt) -> {
            sensors.add(SolomonSensor.createAligned(now.getMillis(), SECONDS_PER_MINUTE, cnt)
                    .withLabel(SolomonSensor.LABEL_CATEGORY, "urlcheck")
                    .withLabel(SolomonSensor.LABEL_SECTION, "data_blocks")
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                    .withLabel("block_type", typeStatePair.getLeft())
                    .withLabel("fetch_state", typeStatePair.getRight()));
        });

        handleCommonMetricsService.handle(sensors, AVERAGE_SENSORS_SIZE);
        monitoringDataStateYDao.update(new MonitoringDataState(MonitoringDataType.LAST_URLCHECK_MONITORING_RUN, "", now));
    }
}
