package ru.yandex.webmaster3.storage;

import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.util.clickhouse2.condition.BoolOpCondition;

/**
 * @author avhaliullin
 */
public abstract class AbstractFilter<I extends Enum<I>> {
    private final I indicator;
    private final Operation operation;
    private final String value;
    private final BoolOpCondition.Operator logicalOperator;

    public AbstractFilter(I indicator, Operation operation, String value) {
        this.indicator = indicator;
        this.operation = operation;
        this.value = value;
        this.logicalOperator = BoolOpCondition.Operator.AND;
    }

    public AbstractFilter(I indicator, Operation operation, String value, BoolOpCondition.Operator logicalOperator) {
        this.indicator = indicator;
        this.operation = operation;
        this.value = value;
        this.logicalOperator = (logicalOperator != null ? logicalOperator : BoolOpCondition.Operator.AND);
    }

    @Description("индикатор, по которому фильтруем")
    public I getIndicator() {
        return indicator;
    }

    @Description("операция фильтрации")
    public Operation getOperation() {
        return operation;
    }

    @Description("значение для фильтрации")
    public String getValue() {
        return value;
    }

    @Description("логический оператор, используется, когда operation = TEXT_MATCH")
    public BoolOpCondition.Operator getLogicalOperator() {
        return logicalOperator;
    }

    public enum Operation {
        LESS_THAN,
        LESS_EQUAL,
        GREATER_THAN,
        GREATER_EQUAL,

        EQUAL,

        TEXT_CONTAINS,
        TEXT_MATCH
    }

    public WebmasterException invalidFilterException() {
        String message = "Bad filter: cannot apply operation " + getOperation() + " to indicator " + getIndicator() + " and value " + getValue();
        return new WebmasterException(message,
                new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), null, null, message));
    }

    public DateTime parseDate() {
        try {
            return ISODateTimeFormat.dateTime().parseDateTime(getValue());
        } catch (IllegalArgumentException e) {
            throw new WebmasterException("Unable to parse date",
                    new WebmasterErrorResponse.IllegalParameterValueResponse(getClass(), null, null), e);
        }
    }
}
