package ru.yandex.webmaster3.storage.abt.dao;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.abt.model.ExperimentScope;
import ru.yandex.webmaster3.storage.abt.model.IExperiment;
import ru.yandex.webmaster3.storage.abt.model.SimpleExperiment;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Insert;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Select;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.Upsert;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/*
 * @author kravchenko99
 * @date 8/21/20
 */
@Repository
public class AbtExperimentYDao extends AbstractYDao {
    private static final String TABLE_NAME = "abt_experiment";
    private static final DataMapper<SimpleExperiment> MAPPER = DataMapper.create(
            F.EXPERIMENT, F.DESCRIPTION, F.EXPERIMENT_SCOPE, F.ACTIVE, SimpleExperiment::new
    );

    static final ValueDataMapper<SimpleExperiment> VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.EXPERIMENT, e -> F.EXPERIMENT.get(e.getName())),
            Pair.of(F.DESCRIPTION, e -> F.DESCRIPTION.get(e.getDescription())),
            Pair.of(F.EXPERIMENT_SCOPE, e -> F.EXPERIMENT_SCOPE.get(e.getScope())),
            Pair.of(F.ACTIVE, e -> F.ACTIVE.get(e.isActive()))
    );

    private static final Map<String, Experiment> EXPERIMENT_MAP = EnumSet.allOf(Experiment.class)
            .stream()
            .collect(Collectors.toMap(Experiment::getName, e -> e, W3Collectors.replacingMerger()));

    public AbtExperimentYDao() {
        super(PREFIX_INTERNAL, TABLE_NAME);
    }

    public IExperiment select(String name) {
        final Select statement = select(MAPPER)
                .where(F.EXPERIMENT.eq(name)).getStatement();
        final SimpleExperiment simpleExperiment = queryOne(statement, MAPPER);
        if (simpleExperiment != null) {
            return simpleExperiment;
        } else {
            return EXPERIMENT_MAP.get(name);
        }
    }

    public void save(String experiment, String description, ExperimentScope scope, boolean active) {
        final Insert insert = insert(F.EXPERIMENT.value(experiment),
                F.DESCRIPTION.value(description),
                F.EXPERIMENT_SCOPE.value(scope),
                F.ACTIVE.value(active));
        execute(insert);
    }

    public void update(IExperiment experiment) {
        update(experiment.getName(), experiment.getDescription(), experiment.getScope(), experiment.isActive());
    }

    public void update(String experiment, String description, ExperimentScope scope, boolean active) {
        final Upsert upsert = upsert(F.EXPERIMENT.value(experiment),
                F.DESCRIPTION.value(description),
                F.EXPERIMENT_SCOPE.value(scope),
                F.ACTIVE.value(active));
        execute(upsert);
    }

    public List<SimpleExperiment> selectAll() {
        return queryForList(select(MAPPER), MAPPER);
    }

    public void insertBatch(List<SimpleExperiment> list) {
        execute(batchInsert(VALUE_MAPPER, list));
    }

    private interface F {
        Field<String> EXPERIMENT = Fields.stringField("experiment");
        Field<String> DESCRIPTION = Fields.stringField("description");
        Field<Boolean> ACTIVE = Fields.boolField("active");
        Field<ExperimentScope> EXPERIMENT_SCOPE = Fields.stringEnumField("experiment_scope", EnumResolver.er(ExperimentScope.class));
    }

}
