package ru.yandex.webmaster3.storage.abt.hash;

import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;


/**
 * @author kravchenko99
 * @date 12/12/20
 */

@Repository
public class AbtHashExperimentYDao extends AbstractYDao {
    private static final String TABLE_NAME = "abt_hash_experiment";

    private static final DataMapper<HashExperimentRecord> MAPPER = DataMapper.create(
            F.EXPERIMENT, F.GROUP, F.HASH_FIELD, F.HASH_FUNCTION, F.MIN_VALUE, F.MAX_VALUE,
            HashExperimentRecord::new
    );
    private static final ValueDataMapper<HashExperimentRecord> VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.EXPERIMENT, e -> F.EXPERIMENT.get(e.getExperiment())),
            Pair.of(F.GROUP, e -> F.GROUP.get(e.getGroup())),
            Pair.of(F.HASH_FIELD, e -> F.HASH_FIELD.get(e.getHashField())),
            Pair.of(F.HASH_FUNCTION, e -> F.HASH_FUNCTION.get(e.getHashFunction())),
            Pair.of(F.MIN_VALUE, e -> F.MIN_VALUE.get(e.getMinValue())),
            Pair.of(F.MAX_VALUE, e -> F.MAX_VALUE.get(e.getMaxValue()))
    );

    public AbtHashExperimentYDao() {
        super(PREFIX_INTERNAL, TABLE_NAME);
    }

    public List<HashExperimentRecord> getRecords() {
        return select(MAPPER).queryForList();
    }

    @Nullable
    public List<HashExperimentRecord> getRecords(String name) {
        return select(MAPPER).where(F.EXPERIMENT.eq(name)).queryForList();
    }

    public void insert(String experiment, String group, HashField hashField, HashFunction hashFunction, Double minValue, Double maxValue) {
        upsert(
                F.EXPERIMENT.value(experiment),
                F.GROUP.value(group),
                F.HASH_FIELD.value(hashField),
                F.HASH_FUNCTION.value(hashFunction),
                F.MIN_VALUE.value(minValue),
                F.MAX_VALUE.value(maxValue)
        ).execute();

    }

    public void delete(HashExperimentRecord record) {
        delete().where(F.EXPERIMENT.eq(record.getExperiment())).and(F.GROUP.eq(record.getGroup())).execute();
    }

    public void batchInsert(List<HashExperimentRecord> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    private interface F {
        Field<String> EXPERIMENT = Fields.stringField("experiment");
        Field<String> GROUP = Fields.stringField("group");
        Field<HashField> HASH_FIELD = Fields.stringEnumField("hash_field", EnumResolver.er(HashField.class));
        Field<HashFunction> HASH_FUNCTION = Fields.stringEnumField("hash_function", EnumResolver.er(HashFunction.class));
        Field<Double> MIN_VALUE = Fields.doubleField("min_value");
        Field<Double> MAX_VALUE = Fields.doubleField("max_value");
    }
}
