package ru.yandex.webmaster3.storage.addurl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.addurl.OwnerRequest;
import ru.yandex.webmaster3.core.addurl.UrlForRecrawl;
import ru.yandex.webmaster3.core.addurl.UrlRecrawlEventLog;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.DailyQuotaUtil;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.UrlUtils;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author tsyplyaev
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Slf4j
public class AddUrlRequestsService implements UserTakeoutDataProvider {
    private static final RetryUtils.RetryPolicy BACKOFF_RETRY_POLICY = RetryUtils.linearBackoff(5, Duration.standardSeconds(30));
    private static final RetryUtils.RetryPolicy INSTANT_RETRY_POLICY = RetryUtils.instantRetry(3);

    private final AddUrlLimitsService addUrlLimitsService;
    private final AddUrlEventsLogsYDao addUrlEventsLogsYDao;
    private final HostOwnerService hostOwnerService;
    private final AddUrlRequestsYDao addUrlRequestsYDao;
    private final AddUrlOwnerRequestsYDao addUrlOwnerRequestsYDao;

    private UrlUtils.CanonizeUrlForRobotWrapper canonizer = new UrlUtils.CanonizeUrlForRobotWrapper();

    public DailyQuotaUtil.QuotaUsage getQuotaUsage(WebmasterHostId hostId, DateTime now) {
        String ownerForStorageQuotaUsage = hostOwnerService.getLongestOwner(hostId);
        return getOwnerQuotaUsage(ownerForStorageQuotaUsage, now);
    }

    public DailyQuotaUtil.QuotaUsage getOwnerQuotaUsage(String owner, DateTime now) {
        try {
            Map<LocalDate, Integer> requestsByDate = addUrlOwnerRequestsYDao.list(owner)
                    .stream()
                    .map(OwnerRequest::getAddDate)
                    .collect(
                            Collectors.groupingBy(
                                    r -> r,
                                    Collectors.summingInt(e -> 1)
                            )
                    );
            TreeMap<LocalDate, Integer> usageMap = new TreeMap<>(requestsByDate);

            //расклеенным оунерам даем минимум
            int maxDailyQuota = addUrlLimitsService.getActualLimit(owner);

            return DailyQuotaUtil.computeRemainingQuotaAddUrl(now.toLocalDate(), usageMap, maxDailyQuota);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to get daily addurl requests count",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), e), e);
        }
    }

    public void increaseQuotaUsage(UrlForRecrawl url) {
        increaseQuotaUsage(List.of(createOwnerRequest(url)));
    }

    public void increaseQuotaUsage(List<OwnerRequest> requests) {
        // на всякий
        var batchPartition = Lists.partition(requests, 1024);
        try {
            for (var b : batchPartition) {
                RetryUtils.execute(INSTANT_RETRY_POLICY, () -> {
                    addUrlOwnerRequestsYDao.addBatch(b);
                });
            }
        } catch (Exception ex) {
            throw new WebmasterException("Unable to add owner requests",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), ex), ex);
        }
    }

    public OwnerRequest createOwnerRequest(UrlForRecrawl url) {
        String ownerForStorageQuotaUsage = hostOwnerService.getLongestOwner(url.getHostId());
        return new OwnerRequest(ownerForStorageQuotaUsage, url.getUrlId(), url.getAddDate());
    }

    public void update(UrlForRecrawl url) {
        try {
            RetryUtils.execute(BACKOFF_RETRY_POLICY, () -> {
                addUrlRequestsYDao.update(url);
            });
        } catch (Exception ex) {
            throw new WebmasterException("Unable to save addurl request",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), ex), ex);
        }
    }

    public void addBatch(List<UrlForRecrawl> batch, String requestId)  {
        // на всякий
        var batchPartition = Lists.partition(batch, 1024);
        try {
            for (var b : batchPartition) {
                RetryUtils.execute(INSTANT_RETRY_POLICY, () -> {
                    addUrlRequestsYDao.addBatch(b, requestId);
                });
            }
        } catch (Exception ex) {
            throw new WebmasterException("Unable to save addurl request",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), ex), ex);
        }
    }

    public void updateBatch(List<UrlForRecrawl> batch) {
        // на всякий
        var batchPartition = Lists.partition(batch, 1024);
        try {
            for (var b : batchPartition) {
                RetryUtils.execute(BACKOFF_RETRY_POLICY, () -> {
                    addUrlRequestsYDao.updateBatch(b);
                });
            }
        } catch (Exception ex) {
            throw new WebmasterException("Unable to save addurl request",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), ex), ex);
        }
    }

    public void add(UrlForRecrawl url, String balancerRequestId) {
        try {
            RetryUtils.execute(INSTANT_RETRY_POLICY, () -> {
                addUrlRequestsYDao.add(url, balancerRequestId);
            });
        } catch (Exception ex) {
            throw new WebmasterException("Unable to save addurl request",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), ex), ex);
        }
    }

    public int count(WebmasterHostId hostId, DateTime fromDate, DateTime toDate) {
        return addUrlRequestsYDao.count(hostId, fromDate, toDate);
    }

    public List<UrlForRecrawl> list(WebmasterHostId hostId, DateTime fromDate, DateTime toDate, int skip, int limit) {
        return addUrlRequestsYDao.list(hostId, fromDate, toDate, skip, limit);
    }

    public UrlForRecrawl get(WebmasterHostId hostId, UUID urlId) {
        return addUrlRequestsYDao.get(hostId, urlId);
    }

    public List<UrlForRecrawl> listUnprocessed(DateTime fromDate, DateTime toDate) {
        var keys = addUrlRequestsYDao.listUnprocessed(fromDate, toDate);
        var keysPartition = Lists.partition(keys, AbstractYDao.YDB_SELECT_ROWS_LIMIT);

        List<UrlForRecrawl> res = new ArrayList<>();
        for (var k : keysPartition) {
            res.addAll(addUrlRequestsYDao.get(k));
        }

        return res;
    }

    public boolean balancerRequestExists(WebmasterHostId hostId, String balancerRequestId) {
        return addUrlRequestsYDao.listRequestIds(hostId, DateTime.now().minus(UrlForRecrawl.STALE_REQUEST_AGE)).stream()
                .filter(Objects::nonNull)
                .anyMatch(req -> req.equals(balancerRequestId));
    }

    public boolean requestExists(WebmasterHostId hostId, String relativeUrl) {
        return !getPendingRequestsForRelativeUrl(hostId, relativeUrl).isEmpty();
    }

    public List<UrlForRecrawl> getPendingRequestsForRelativeUrl(WebmasterHostId hostId, String relativeUrl) {
        try {
            return RetryUtils.query(BACKOFF_RETRY_POLICY, () -> {
                return addUrlRequestsYDao.list(hostId, DateTime.now().minus(UrlForRecrawl.STALE_REQUEST_AGE)).stream()
                        .filter(req -> !req.getState().isTerminal())
                        .filter(req -> req.getRelativeUrl().equals(relativeUrl))
                        .collect(Collectors.toList());

            });
        } catch (Exception e) {
            throw new WebmasterException("Unable to find request",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), e), e);
        }
    }

    public List<UrlForRecrawl> getPendingRequests(WebmasterHostId hostId, String samovarUrl) {
        try {
            return RetryUtils.query(BACKOFF_RETRY_POLICY, () -> {
                return addUrlRequestsYDao.list(hostId, DateTime.now().minus(UrlForRecrawl.STALE_REQUEST_AGE)).stream()
                        .filter(req -> !req.getState().isTerminal())
                        .filter(req -> {
                            String canonicalUrl = canonizer.canonizeUrlForRobot(req.getFullUrl());
                            if (canonicalUrl == null) {
                                canonicalUrl = req.getFullUrl();
                            }

                            return canonicalUrl.equals(samovarUrl);
                        })
                        .collect(Collectors.toList());
            });

        } catch (Exception e) {
            throw new WebmasterException("Error reading from Ydv",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), e), e);
        }
    }

    @Nullable
    public String toRelativeUrlWithVerification(WebmasterHostId hostId, String url) {
        try {
            // WMC-6221 Костыль аналогичный https://a.yandex-team.ru/arc/trunk/arcadia/yweb/robot/zoracl/lib/ilogger.cpp?blame=true&rev=3873578#L220
            String relativeUrl = IdUtils.toRelativeUrl(hostId, url);
            if (relativeUrl != null && !relativeUrl.contains("\t")) {
                return relativeUrl;
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error("Unable to parse url: {}", url, e);
            return null;
        }
    }

    public void saveLogs(UrlRecrawlEventLog urlRecrawlLog) {
        try {
            addUrlEventsLogsYDao.add(urlRecrawlLog);
        } catch (Exception ex) {
            // игнорируем, потеря записей в логе не страшна
            log.error("Error saving event log", ex);
        }
    }

    public void saveLogs(List<UrlRecrawlEventLog> batch) {
        var batchPartition = Lists.partition(batch, 1024);
        try {
            for (var b : batchPartition) {
                RetryUtils.execute(BACKOFF_RETRY_POLICY, () -> {
                    addUrlEventsLogsYDao.addBatch(b);
                });
            }
        } catch (Exception ex) {
            // игнорируем, потеря записей в логе не страшна
            log.error("Error saving event log", ex);
        }
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        addUrlEventsLogsYDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                addUrlEventsLogsYDao.getTablePath()
        );
    }

    @VisibleForTesting
    public void setCanonizeUrlForRobotWrapper(UrlUtils.CanonizeUrlForRobotWrapper canonizer) {
        this.canonizer = canonizer;
    }
}
