package ru.yandex.webmaster3.storage.admin.dao;

import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.admin.SelfVerificationRecord;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.SelectQuery;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.QueryBuilder;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author kravchenko99
 * @date 12/22/20
 */

@Repository
public class SelfHostVerificationsYDao extends AbstractYDao implements UserTakeoutDataProvider {

    private static final String TABLE_NAME = "self_host_verifications";
    private static final String LOGIN_INDEX = "admin_user_id_index";

    public SelfHostVerificationsYDao() {
        super(PREFIX_ADMIN, TABLE_NAME);
    }

    public void addVerificationRecord(SelfVerificationRecord record) {
        upsert(
                F.ADMIN_USER_ID.value(record.getAdminUserId()),
                F.USER_ID.value(record.getUserId()),
                F.HOST_ID.value(record.getHostId()),
                F.CREATED_DATE.value(record.getCreatedDate())
        ).execute();
    }

    public SelfVerificationRecord getVerificationRecord(WebmasterHostId hostId, long userId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .and(F.USER_ID.eq(userId))
                .queryOne();
    }

    public List<SelfVerificationRecord> listSelfVerificationsForUser(long adminUserId) {
        return select(MAPPER)
                .secondaryIndex(LOGIN_INDEX)
                .where(F.ADMIN_USER_ID.eq(adminUserId))
                .queryForList(
                        Pair.of(F.USER_ID, SelfVerificationRecord::getUserId),
                        Pair.of(F.HOST_ID, SelfVerificationRecord::getHostId)
                );
    }

    public void deleteRecord(WebmasterHostId hostId, long userId) {
        delete()
                .where(F.HOST_ID.eq(hostId))
                .and(F.USER_ID.eq(userId))
                .execute();
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                getTablePath()
        );
    }

    private static final DataMapper<SelfVerificationRecord> MAPPER = DataMapper.create(
            F.ADMIN_USER_ID, F.USER_ID, F.HOST_ID, F.CREATED_DATE, SelfVerificationRecord::new
    );

    public void batchInsert(List<SelfVerificationRecord> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    private static final ValueDataMapper<SelfVerificationRecord> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, SelfVerificationRecord::getUserId),
            Pair.of(F.HOST_ID, SelfVerificationRecord::getHostId),
            Pair.of(F.ADMIN_USER_ID, SelfVerificationRecord::getAdminUserId),
            Pair.of(F.CREATED_DATE, SelfVerificationRecord::getCreatedDate)
    );

    private static class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<Long> USER_ID = Fields.longField("user_id");
        static final Field<Long> ADMIN_USER_ID = Fields.longField("admin_user_id");
        static final Field<DateTime> CREATED_DATE = Fields.jodaDateTimeField("created_date");
    }
}
