package ru.yandex.webmaster3.storage.admin.dao;

import java.util.List;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.admin.UserRoleRecord;
import ru.yandex.webmaster3.storage.admin.security.Role;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;


/**
 * @author kravchenko99
 * @date 12/22/20
 */

@Repository
public class UserRolesYDao extends AbstractYDao implements UserTakeoutDataProvider {
    private static final String TABLE_NAME = "user_roles";

    public UserRolesYDao() {
        super(PREFIX_ADMIN, TABLE_NAME);
    }

    public List<UserRoleRecord> getUserRoles(long userId) {
        return select(MAPPER)
                .where(F.USER_ID.eq(userId))
                .queryForList();
    }

    public List<UserRoleRecord> getAllUsersRoles() {
        return select(MAPPER)
                .queryForList(Pair.of(F.USER_ID, UserRoleRecord::getUserId));

    }

    public void addRole(long userId, Role role, String requestedLogin) {
        upsert(
                F.USER_ID.value(userId),
                F.ROLE_ID.value(role),
                F.REQUESTED_LOGIN.value(requestedLogin)
        ).execute();
    }

    public void removeRole(long userId, Role role) {
        delete()
                .where(F.USER_ID.eq(userId))
                .and(F.ROLE_ID.eq(role))
                .execute();
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private static final DataMapper<UserRoleRecord> MAPPER = DataMapper.create(
            F.USER_ID, F.ROLE_ID, F.REQUESTED_LOGIN, UserRoleRecord::new
    );


    public void batchInsert(List<UserRoleRecord> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                getTablePath()
        );
    }

    private static final ValueDataMapper<UserRoleRecord> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, UserRoleRecord::getUserId),
            Pair.of(F.ROLE_ID, UserRoleRecord::getRole),
            Pair.of(F.REQUESTED_LOGIN, UserRoleRecord::getRequestedLogin)
    );

    private static class F {
        static final Field<Long> USER_ID = Fields.longField("user_id");
        static final Field<Role> ROLE_ID = Fields.stringEnumField("role_id", Role.R);
        static final Field<String> REQUESTED_LOGIN = Fields.stringField("requested_login");
    }
}
