package ru.yandex.webmaster3.storage.antispam.threats.dao;

import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.proto.Antispam;
import ru.yandex.webmaster3.storage.antispam.threats.data.HostThreatInfo;
import ru.yandex.webmaster3.storage.antispam.threats.data.ThreatPropagationMode;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * DAO для угроз
 * Created by Oleg Bazdyrev on 09/08/2017.
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class OwnerThreatsCHDao extends AbstractClickhouseDao {

    private final HostOwnerService hostOwnerService;
    @Setter
    private TableProvider tableStorage;

    /**
     * Возвращает угрозы и даты их обнаружения (ActualSince)
     */
    @NotNull
    public HostThreats getHostWithThreats(WebmasterHostId hostId) {
        Preconditions.checkNotNull(hostId);

        return getHostWithThreats(List.of(hostId)).get(hostId);
    }

    @NotNull
    public Map<WebmasterHostId, HostThreats> getHostWithThreats(Collection<WebmasterHostId> hostIds) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.OWNER_THREATS);

        Map<String, List<WebmasterHostId>> hostsByOwner = hostIds.stream()
                .collect(Collectors.groupingBy(x -> hostOwnerService.getHostOwner(x.getPunycodeHostname())));


        Where st = QueryBuilder.select(F.OWNER, F.HOST, F.ACTUAL_SINCE, F.THREATS).from(table.getClickhouseFullName())
                .where(QueryBuilder.in(F.OWNER, hostsByOwner.keySet()));

        Mapper mapper = new Mapper(hostsByOwner);
        // берем запросы для всех хостов и групируем по hostId
        var hostThreatsByHost =
                getClickhouseServer().queryAll(st.toString(), mapper).stream()
                        .flatMap(Collection::stream)
                        .collect(Collectors.groupingBy(HostThreats::getHostId));

        Map<WebmasterHostId, HostThreats> threatsByHost = new HashMap<>();
        for (var entry : hostThreatsByHost.entrySet()) {
            WebmasterHostId hostId = entry.getKey();
            var threatsWithActualSince = entry.getValue();
            HostThreats hostThreats = processHostThreats(hostId, threatsWithActualSince);
            threatsByHost.put(hostId, hostThreats);
        }
        // докладываем пустые
        hostIds.forEach(hostId -> threatsByHost.putIfAbsent(hostId, HostThreats.empty(hostId)));

        return threatsByHost;
    }

    @NotNull
    private HostThreats processHostThreats(WebmasterHostId hostId,
                                           List<HostThreats> threatsWithActualSince) {
        DateTime minActualSince = null;
        List<HostThreatInfo> allThreats = new ArrayList<>();
        for (var pair : threatsWithActualSince) {
            minActualSince = ObjectUtils.min(minActualSince, pair.getActualSince());
            allThreats.addAll(pair.getThreats());
        }

        return new HostThreats(hostId, minActualSince, allThreats);
    }

    @AllArgsConstructor
    private static final class Mapper implements Function<CHRow, List<HostThreats>> {
        Map<String, List<WebmasterHostId>> hostsByOwner;

        @Override
        public List<HostThreats> apply(CHRow chRow) {
            String owner = chRow.getString(F.OWNER);
            List<WebmasterHostId> hosts = hostsByOwner.get(owner);

            return hosts.stream().map(host -> applyInternal(chRow, host))
                    .collect(Collectors.toList());
        }

        public static HostThreats applyInternal(CHRow chRow, WebmasterHostId hostId) {
            try {
                String requestHost = hostId.getPunycodeHostname();
                String host = chRow.getString(F.HOST);
                DateTime actualSince = new DateTime(chRow.getLongUnsafe(F.ACTUAL_SINCE));
                byte[] bytes = chRow.getBytes(F.THREATS);
                Antispam.HostThreats hostThreats = Antispam.HostThreats.parseFrom(Base64.getDecoder().decode(bytes));
                List<HostThreatInfo> threats = new ArrayList<>();
                // в зависимости от propagation-mode оставляем те или иные угрозы
                for (Antispam.ThreatInfo rawThreat : hostThreats.getThreatsList()) {
                    boolean allowedToRecheck = WwwUtil.equalsIgnoreWww(requestHost, host);
                    HostThreatInfo threatInfo =
                            HostThreatInfo.fromProto(rawThreat).withAllowedToRecheck(allowedToRecheck);
                    if (threatInfo.getPropagationMode() == ThreatPropagationMode.OWNER
                            || threatInfo.getPropagationMode() == ThreatPropagationMode.SUBDOMAINS
                            && (requestHost.equals(host) || requestHost.endsWith("." + host))) {
                        threats.add(threatInfo);
                    }
                }
                return new HostThreats(hostId, actualSince, threats);
            } catch (InvalidProtocolBufferException e) {
                throw new WebmasterException("Error reading protobuf threats from clickhouse",
                        new WebmasterErrorResponse.ClickhouseErrorResponse(OwnerThreatsCHDao.class, null, e), e);
            }
        }
    }

    private interface F {
        String OWNER = "owner";
        String HOST = "host";
        String ACTUAL_SINCE = "actual_since";
        String THREATS = "threats";
    }

    @Value
    public static class HostThreats {
        WebmasterHostId hostId;
        DateTime actualSince;
        List<HostThreatInfo> threats;

        public static HostThreats empty(WebmasterHostId hostId) {
            return new HostThreats(hostId, null, new ArrayList<>());
        }
    }

}
