package ru.yandex.webmaster3.storage.avatars;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.xerces.util.SecurityManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.common.util.Su;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.avatars.UploadPictureResult.InvalidPictureErrorCode;
import ru.yandex.webmaster3.storage.turbo.logo.TurboLogoData;

/**
 * Created by ifilippov5 on 13.07.17.
 */
public class AvatarImageStoreService extends AbstractExternalAPIService {
    private static final Logger log = LoggerFactory.getLogger(AvatarImageStoreService.class);

    private static final int SOCKET_TIMEOUT = 20000;
    private static final int CONNECTION_TIMEOUT = 20000;

    private static final int HTTP_CODE_IMAGE_IS_TOO_SMALL = 415;
    private static final int HTTP_CODE_RPS_RATE_LIMITER = 429;
    private static final int HTTP_CODE_ERROR_IN_DOWNLOAD_BY_URL = 434;

    private static final Predicate<Exception> USER_ERROR_PREDICATE = userExceptions(AvatarsException.class);

    private String indexer;
    private String viewer;
    private String namespace;
    private Set<String> viewImageSizes;

    private final CloseableHttpClient client =
            HttpClientBuilder.create().setDefaultRequestConfig(
                    RequestConfig.custom()
                            .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                            .setSocketTimeout(SOCKET_TIMEOUT)
                            .build()
            ).setConnectionReuseStrategy(DefaultConnectionReuseStrategy.INSTANCE)
                    .setRetryHandler(new DefaultHttpRequestRetryHandler(5, true))
                    .build();

    @ExternalDependencyMethod("upload-url")
    public UploadPictureResult  uploadPicture(String url, Integer ttlInDays) throws AvatarsException {
        return trackQuery(new JavaMethodWitness() {}, USER_ERROR_PREDICATE, () -> {
            int counter = 0;
            String request;
            if (ttlInDays != null) {
                request = indexer + "/put-" + namespace + "/?expired=" + String.valueOf(ttlInDays) + "d&url=" + URLEncodeUtil.urlEncode(url);
            } else {
                request = indexer + "/put-" + namespace + "/?url=" + URLEncodeUtil.urlEncode(url);
            }
            HttpGet getJson = new HttpGet(request);
            while (true) {
                try (CloseableHttpResponse response = client.execute(getJson)) {
                    int code = response.getStatusLine().getStatusCode();
                    String responseStr;
                    if (code != HttpStatus.SC_OK) {
                        log.warn("Not 200 answer from avatars code is " + code);
                        responseStr = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                        log.debug("Upload response:\n{}", responseStr);
                    }
                    if (code == HttpStatus.SC_OK) {
                        log.debug("Ok request");
                        return parsePicturesFromResponse(response, false);
                    } else if (code == HTTP_CODE_IMAGE_IS_TOO_SMALL) {
                        return UploadPictureResult.builder().invalidPictureErrorCode(InvalidPictureErrorCode.IMAGE_IS_TOO_SMALL).build();
                    } else if (code == HTTP_CODE_ERROR_IN_DOWNLOAD_BY_URL) {
                        return UploadPictureResult.builder().invalidPictureErrorCode(InvalidPictureErrorCode.UNABLE_TO_DOWNLOAD_IMAGE_BY_URL).build();
                    } else if (code == HTTP_CODE_RPS_RATE_LIMITER) {
                        if (++counter < 5) {
                            Thread.sleep(1000);
                        } else {
                            throw new AvatarsException("Not 200 answer from avatars 5 times in row code is " + code);
                        }
                    } else if (code / 100 == 4) {
                        log.error("Avatars service returns code {}", code);
                        throw new AvatarsException("Not 200 answer from avatars code is " + code);
                    } else {
                        log.error("Avatars service returns code {}", code);
                        throw new WebmasterException("Failed to upload turbo logo by url. Code: " + code,
                                new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), null));
                    }
                } catch (InterruptedException e) {
                    log.error("Request Interrupted", e);
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e);
                } catch (IOException e) {
                    throw new WebmasterException("Failed to upload turbo logo by url",
                            new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), e));
                }
            }
        });
    }

    @ExternalDependencyMethod("upload-content")
    public UploadPictureResult uploadPicture(String fileName, byte[] imageByteContent)
            throws AvatarsException {

        // проверим, не svg ли перед нами (отключено, пока не поддержано на стороне турбо)
        final boolean isSvg = isSvg(imageByteContent);
        HttpPost postJson = new HttpPost(indexer + "/put-" + namespace);
        postJson.setEntity(MultipartEntityBuilder.create()
                .addPart(isSvg ? "svg" : "file", new ByteArrayBody(imageByteContent, fileName)).build());

        return trackQuery(new JavaMethodWitness() {}, USER_ERROR_PREDICATE, () -> {
            try (CloseableHttpResponse response = client.execute(postJson)) {
                int code = response.getStatusLine().getStatusCode();
                if (code != HttpStatus.SC_OK) {
                    String responseStr = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                    log.debug("Upload response:\n{}", responseStr);
                }
                if (code == HttpStatus.SC_OK) {
                    log.debug("Ok request");
                    return parsePicturesFromResponse(response, isSvg);
                } else if (code == HTTP_CODE_IMAGE_IS_TOO_SMALL) {
                    return UploadPictureResult.builder().invalidPictureErrorCode(InvalidPictureErrorCode.IMAGE_IS_TOO_SMALL).build();
                } else if (code / 100 == 4) {
                    log.error("Avatars service returns code {}", code);
                    throw new AvatarsException("Not 200 answer from avatars code is " + code);
                } else {
                    log.error("Avatars service returns code {}", code);
                    throw new WebmasterException("Failed to upload turbo logo. Code: " + code,
                            new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), null));
                }
            } catch (IOException e) {
                throw new WebmasterException("Failed to upload picture", new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), e), e);
            }
        });
    }

    @ExternalDependencyMethod("get-imageinfo")
    public UploadPictureResult getImageInfo(String logoId, boolean svg) {
        return trackQuery(new JavaMethodWitness() {}, USER_ERROR_PREDICATE, () -> {
            Pair<String, Long> pair = TurboLogoData.parseFrontLogoId(logoId);
            HttpGet get = new HttpGet(indexer + "/getimageinfo-" + namespace + "/" + pair.getRight() + "/" + pair.getLeft());
            try (CloseableHttpResponse response = client.execute(get)) {
                int code = response.getStatusLine().getStatusCode();
                if (code != HttpStatus.SC_OK) {
                    String responseStr = EntityUtils.toString(response.getEntity(), StandardCharsets.UTF_8);
                    log.debug("Upload response:\n{}", responseStr);
                }
                if (code == HttpStatus.SC_OK) {
                    log.debug("Ok request");
                    return parsePicturesFromResponse(response, svg);
                } else {
                    log.error("Avatars service returns code {}", code);
                    throw new WebmasterException("Failed to get image info. Code: " + code,
                            new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), null));
                }
            } catch (IOException e) {
                throw new WebmasterException("Failed to get image info", new WebmasterErrorResponse.AvatarsErrorResponse(getClass(), e), e);
            }
        });
    }

    private UploadPictureResult parsePicturesFromResponse(CloseableHttpResponse response, boolean isSvg) throws IOException {
        var resultBuilder = UploadPictureResult.builder().isSvg(isSvg);
        ObjectNode root = (ObjectNode) JsonMapping.OM.readTree(response.getEntity().getContent());
        Set<String> sizes = isSvg ? Collections.singleton("svg") : viewImageSizes;
        ObjectNode sizesObject = (ObjectNode) root.get(isSvg ? "extra" : "sizes");
        List<AvatarPicture> avatarsImages = new ArrayList<>();
        for (String size : sizes) {
            String name = extractImageName(sizesObject.get(size).get("path").asText());
            long groupId = root.get("group-id").asLong();
            avatarsImages.add(new AvatarPicture(namespace, name, groupId, size));
        }
        resultBuilder.pictures(avatarsImages);
        // width/height
        if (!isSvg) {
            ObjectNode origNode = (ObjectNode) root.get("sizes").get("orig");
            resultBuilder.width(origNode.get("width").asInt());
            resultBuilder.height(origNode.get("height").asInt());
        }
        return resultBuilder.build();
    }

    public boolean deletePicture(AvatarPicture picture) throws IOException {

        String filename = picture.getFilename();
        String group = Long.toString(picture.getGroupId());

        return deletePicture(filename, group);
    }

    @ExternalDependencyMethod("delete")
    public boolean deletePicture(String filename, String group) throws IOException {
        return trackQuery(new JavaMethodWitness() {}, USER_ERROR_PREDICATE, () -> {
            String deleteUrl = Su.join(new String[]{indexer, "delete-" + namespace, group, filename}, "/");
            log.debug("Filename: {}, groupId: {}", filename, group);
            try (CloseableHttpClient client = HttpClientBuilder.create()
                    .setRetryHandler(new DefaultHttpRequestRetryHandler(5, true)).build()) {

                HttpGet getJson = new HttpGet(deleteUrl);
                try (CloseableHttpResponse response = client.execute(getJson)) {
                    int code = response.getStatusLine().getStatusCode();
                    boolean deleted = (code == HttpStatus.SC_OK || code == HttpStatus.SC_NOT_FOUND);
                    if (code == HttpStatus.SC_NOT_FOUND) {
                        log.info("File already deleted");
                    }
                    if (!deleted) {
                        log.info("File " + group + "/" + filename + " not deleted code is: " + response.getStatusLine().getStatusCode());
                    }
                    if (!deleted && code != HttpStatus.SC_ACCEPTED) {
                        throw new RuntimeException("Internal error during delete");
                    }
                    return deleted;
                }
            } catch (ClientProtocolException e) {
                log.error("Wrong protocol in avatars", e);
                throw e;
            } catch (IOException e) {
                log.error("IOException in avatars", e);
                throw e;
            }
        });
    }

    public String extractGroupId(String avatarsPath) {
        String[] parts = avatarsPath.split("/");
        if (parts.length > 3) {
            return parts[2];
        } else {
            return null;
        }
    }

    public String extractImageName(String avatarsPath) {
        String[] parts = avatarsPath.split("/");
        if (parts.length > 3) {
            return parts[3];
        } else {
            return null;
        }
    }

    public String getPictureUrl(AvatarPicture picture) {
        return Su.join(new String[]{viewer, "get-" + namespace, String.valueOf(picture.getGroupId()), picture.getFilename(), picture.getSize()}, "/");
    }

    public String getPictureUrl(AvatarPicture picture, String size) {
        return Su.join(new String[]{viewer, "get-" + namespace, String.valueOf(picture.getGroupId()), picture.getFilename(), size}, "/");
    }

    public AvatarPicture getPicture(Set<AvatarPicture> avatarPictures, String imageSize) {
        for (AvatarPicture picture : avatarPictures) {
            if (imageSize.equals(picture.getSize())) {
                return picture;
            }
        }
        return null;
    }

    /**
     * Попытка определить, не svg ли перед нами
     *
     * @param imageContent
     * @return
     */
    private static boolean isSvg(byte[] imageContent) {
        try {
            final SAXParserFactory factory = SAXParserFactory.newInstance();
            factory.setNamespaceAware(false);
            factory.setValidating(false);
            SAXParser parser = factory.newSAXParser();
            XMLReader reader = parser.getXMLReader();
            reader.setFeature("http://apache.org/xml/features/validation/schema/augment-psvi", false);
            reader.setFeature("http://apache.org/xml/features/nonvalidating/load-dtd-grammar", false);
            reader.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);
            reader.setFeature("http://xml.org/sax/features/external-parameter-entities", false);
            reader.setFeature("http://xml.org/sax/features/external-general-entities", false);
            reader.setFeature("http://xml.org/sax/features/validation", false);
            reader.setProperty("http://apache.org/xml/properties/security-manager", new SecurityManager());
            DefaultHandler rootTagHandler = new DefaultHandler() {
                @Override
                public void startElement(String uri, String localName, String qName, Attributes attributes)
                        throws SAXException {
                    if ("svg".equals(qName)) {
                        throw new SvgException();
                    }
                    throw new SAXException();
                }
            };
            ByteArrayInputStream bais = new ByteArrayInputStream(imageContent);
            parser.parse(bais, rootTagHandler);
        } catch (SvgException e) {
            return true;
        } catch (Exception e) {
            return false;
        }
        return false;
    }

    @Required
    public void setIndexer(String indexer) {
        this.indexer = indexer;
    }

    @Required
    public void setViewer(String viewer) {
        this.viewer = viewer;
    }

    @Required
    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Required
    public void setViewImageSizes(Set<String> viewImageSizes) {
        this.viewImageSizes = viewImageSizes;
    }

    private static class SvgException extends SAXException {

    }
}