package ru.yandex.webmaster3.storage.cache.dao;

import lombok.Data;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;


/**
 * @author leonidrom
 */
@Repository
public class WidgetCacheYDao extends AbstractYDao {
    private static final Duration TTL = Duration.standardMinutes(10);
    private static final String TABLE_NAME = "widget_cache";

    public WidgetCacheYDao() {
        super(PREFIX_CACHE, TABLE_NAME);
    }

    public void saveData(WebmasterHostId hostId, byte[] data) {
        upsert(
                F.HOST_ID.value(hostId),
                F.DATA.value(data),
                F.UPDATE_DATE.value(Instant.now()))
        .execute();
    }

    @Nullable
    public byte[] getData(WebmasterHostId hostId) {
        var rec = select(MAPPER).where(F.HOST_ID.eq(hostId)).queryOne();
        if (rec == null || rec.update_date.plus(TTL).isBeforeNow()) {
            return null;
        }

        return rec.data;
    }

    @Data
    private static class Record {
        final WebmasterHostId hostId;
        final byte[] data;
        final Instant update_date;
    }

    private static final DataMapper<Record> MAPPER = DataMapper.create(F.HOST_ID, F.DATA, F.UPDATE_DATE, Record::new);

    private static class F {
        private static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        private static final Field<byte[]> DATA = Fields.byteArrayField("data");
        private static final Field<Instant> UPDATE_DATE = Fields.jodaInstantField("update_date");
    }
}
