package ru.yandex.webmaster3.storage.checklist.dao;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.checklist.data.SiteProblemRecheckInfo;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author avhaliullin
 */
@Repository
@Slf4j
public class SiteProblemsRecheckYDao extends AbstractYDao {
    public SiteProblemsRecheckYDao() {
        super(PREFIX_WEBMASTER3, "site_problems_recheck");
    }

    public List<SiteProblemRecheckInfo> getRecheckInfoForHost(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId)).queryForList();
    }

    public SiteProblemRecheckInfo getRecheckInfo(WebmasterHostId hostId, SiteProblemTypeEnum problemType) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .and(F.PROBLEM_TYPE.eq(problemType))
                .queryOne();
    }

    public void updateRecheckState(WebmasterHostId hostId, SiteProblemTypeEnum problemType, boolean recheckRequested) {
        updateRecheckState(hostId, problemType, recheckRequested, DateTime.now());
    }

    public void updateRecheckState(WebmasterHostId hostId, SiteProblemTypeEnum problemType, boolean recheckRequested, DateTime dateTime) {
        upsert(
                F.HOST_ID.value(hostId),
                F.PROBLEM_TYPE.value(problemType),
                F.RECHECK_REQUESTED.value(recheckRequested),
                F.RECHECK_FAILED.value(false),
                F.REQUEST_DATE.value(dateTime)
        ).execute();
    }

    public void updateRecheckFailed(WebmasterHostId hostId, SiteProblemTypeEnum problemType, boolean recheckFailed) {
        update(F.RECHECK_FAILED.set(recheckFailed))
                .where(F.HOST_ID.eq(hostId)).and(F.PROBLEM_TYPE.eq(problemType))
                .execute();
    }

    public void foreach(Consumer<SiteProblemRecheckInfo> consumer) {
        streamReader(MAPPER, consumer);
    }

    public void delete(Collection<WebmasterHostId> hostIds, Collection<SiteProblemTypeEnum> problems) {
        if (hostIds.isEmpty() || problems.isEmpty()) {
            return;
        }

        List<Pair<WebmasterHostId, SiteProblemTypeEnum>> items = new ArrayList<>();

        for (var hostId: hostIds) {
            for (var problem: problems) {
                items.add(Pair.of(hostId, problem));
            }
        }

        batchDelete(DELETE_DATA_MAPPER, items);
    }

    private static final DataMapper<SiteProblemRecheckInfo> MAPPER = DataMapper.create(
            F.PROBLEM_TYPE, F.HOST_ID, F.RECHECK_REQUESTED, F.RECHECK_FAILED, F.REQUEST_DATE,
            (problemType, hostId, recheckRequested, recheckFailed, requestDate) -> {
                if (problemType == null) {
                    return null;
                }

                return new SiteProblemRecheckInfo(hostId, problemType, recheckRequested, recheckFailed, requestDate);
            }
    );

    private static final ValueDataMapper<Pair<WebmasterHostId, SiteProblemTypeEnum>> DELETE_DATA_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, Pair::getKey),
            Pair.of(F.PROBLEM_TYPE, Pair::getValue)
    );

    private static class F {
        public static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        public static final Field<SiteProblemTypeEnum> PROBLEM_TYPE = Fields.intEnumField("problem_type", SiteProblemTypeEnum.R);
        public static final Field<Boolean> RECHECK_REQUESTED = Fields.boolField("recheck_requested").withDefault(false);//true, если нажали хотя бы раз
        public static final Field<DateTime> REQUEST_DATE = Fields.jodaDateTimeField("request_date");
        public static final Field<Boolean> RECHECK_FAILED = Fields.boolField("recheck_failed").withDefault(false);
    }
}
