package ru.yandex.webmaster3.storage.checklist.dao;

import java.util.Collection;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.core.type.TypeReference;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.checklist.data.YaBrowserBadAdFormat;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * Created by Oleg Bazdyrev on 11/12/2017.
 */

@Repository
public class YaBrowserBadAdRecheckQueueYDao extends AbstractYDao {
    public static final TypeReference<Set<String>> TYPE_REFERENCE = new TypeReference<>() {
    };

    public static final String TABLE_NAME = "yabrowser_badad_recheck_queue";

    private static final DataMapper<RecheckRecord> MAPPER = DataMapper.create(F.ID, F.HOST_ID, F.DATE, F.FORMATS,
            RecheckRecord::new);

    public YaBrowserBadAdRecheckQueueYDao() {
        super(PREFIX_WEBMASTER3, TABLE_NAME);
    }

    public void addRecord(WebmasterHostId hostId, Set<YaBrowserBadAdFormat> formats) {
        insert(F.ID.value(UUIDs.timeBased()),
                F.HOST_ID.value(hostId),
                F.DATE.value(DateTime.now()),
                F.FORMATS.value(formats.stream().map(YaBrowserBadAdFormat::getRawId).collect(Collectors.toSet())));
    }

    public void deleteRecords(Collection<UUID> ids) {

        execute(delete().where(F.ID.in(ids)).getStatement());
    }

    public void forEach(Consumer<RecheckRecord> consumer) {
        streamReader(MAPPER, consumer);
    }

    private interface F {
        Field<UUID> ID = Fields.uuidField("id");
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<DateTime> DATE = Fields.jodaDateTimeField("date");
        Field<Set<String>> FORMATS = Fields.jsonField("formats", TYPE_REFERENCE);
    }

    public static final class RecheckRecord {

        private final UUID id;
        private final WebmasterHostId hostId;
        private final DateTime date;
        private final Set<String> formats;

        public RecheckRecord(UUID id, WebmasterHostId hostId, DateTime date, Set<String> formats) {
            this.id = id;
            this.hostId = hostId;
            this.date = date;
            this.formats = formats;
        }

        public UUID getId() {
            return id;
        }

        public WebmasterHostId getHostId() {
            return hostId;
        }

        public DateTime getDate() {
            return date;
        }

        public Set<String> getFormats() {
            return formats;
        }
    }
}
