package ru.yandex.webmaster3.storage.checklist.data;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemSeverityEnum;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.checklist.util.SiteProblemWeightUtil;

/**
 * @author avhaliullin
 */
public class AbstractProblemInfo extends ProblemSignal {
    private final WebmasterHostId hostId;
    @Nullable
    private final DateTime actualSince;
    private final int weight;

    public AbstractProblemInfo(WebmasterHostId hostId, @Nullable SiteProblemContent content, SiteProblemTypeEnum problemType,
                               @Nullable DateTime actualSince, SiteProblemState state, DateTime lastUpdate, int weight) {
        super(content, problemType, state, lastUpdate);
        this.hostId = hostId;
        this.actualSince = actualSince;
        this.weight = weight;
    }

    public AbstractProblemInfo(WebmasterHostId hostId, SiteProblemTypeEnum problemType) {
        this(hostId, null, problemType, null,
                problemType.getDefaultState(), null, SiteProblemWeightUtil.computeWeight(problemType));
    }

    public SiteProblemState getUserVisibleState(boolean isMainMirror) {
        SiteProblemTypeEnum problemType = getProblemType();
        if (problemType.isDisabled() || (problemType.isOnlyForMainMirror() && !isMainMirror)) {
            return SiteProblemState.NOT_APPLICABLE;
        } else {
            return getState();
        }
    }

    public AbstractProblemInfo withState(SiteProblemState state) {
        return new AbstractProblemInfo(hostId, getContent(), getProblemType(), getActualSince(),
                state, getLastUpdate(), getWeight());
    }

    public WebmasterHostId getHostId() {
        return hostId;
    }

    @Nullable
    public DateTime getActualSince() {
        return actualSince;
    }

    public int getWeight() {
        return weight;
    }

    @JsonIgnore
    public boolean isRecommendation() {
        return getProblemType().getSeverity() == SiteProblemSeverityEnum.RECOMMENDATION;
    }
}
