package ru.yandex.webmaster3.storage.checklist.data;

import lombok.ToString;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;

import java.util.function.Function;
import java.util.function.Supplier;

/**
 * @author avhaliullin
 */
@ToString(callSuper = true)
public class ProblemSignal extends ProblemStateInfo {
    @Nullable
    private final SiteProblemContent content;
    private final DateTime lastUpdate;

    protected ProblemSignal(@Nullable SiteProblemContent content, SiteProblemTypeEnum problemType,
                            SiteProblemState state, DateTime lastUpdate) {
        super(problemType, state);
        this.content = content;
        this.lastUpdate = lastUpdate;
    }

    public ProblemSignal(SiteProblemContent content, DateTime lastUpdate) {
        this(content, content.getProblemType(), SiteProblemState.PRESENT, lastUpdate);
    }

    public ProblemSignal(SiteProblemTypeEnum problemType, SiteProblemState state, DateTime lastUpdate) {
        this(null, problemType, state, lastUpdate);
    }

    @Nullable
    public SiteProblemContent getContent() {
        return content;
    }

    public DateTime getLastUpdate() {
        return lastUpdate;
    }

    // Функциональные методы для матчинга состояний - можно обезопаситься от NPE и warning'ов про nullable getContent.

    public <T> T fold(Function<SiteProblemContent, T> ifPresent, Supplier<T> ifNotPresent) {
        return fold(ifPresent, ifNotPresent, ifNotPresent);
    }

    public <T> T fold(Function<SiteProblemContent, T> ifPresent, Supplier<T> ifAbsent, Supplier<T> ifUndefined) {
        switch (getState()) {
            case PRESENT:
                if (content == null) {
                    throw new WebmasterException("Broken invariant: ProblemSignal problem is present, but problem content is null. Class: " + this.getClass().getSimpleName(),
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), ""));
                }
                return ifPresent.apply(content);
            case ABSENT:
                return ifAbsent.get();
            case UNDEFINED:
                return ifUndefined.get();
            default:
                throw new RuntimeException("Unknown site problem state: " + getState());
        }
    }

    public static ProblemSignal createAbsent(SiteProblemTypeEnum problemType, DateTime lastUpdate) {
        return new ProblemSignal(problemType, SiteProblemState.ABSENT, lastUpdate);
    }

    public static ProblemSignal createPresent(SiteProblemContent content, DateTime lastUpdate) {
        return new ProblemSignal(content, lastUpdate);
    }

    public static ProblemSignal createUndefined(SiteProblemTypeEnum problemType, DateTime lastUpdate) {
        return new ProblemSignal(problemType, SiteProblemState.UNDEFINED, lastUpdate);
    }
}