package ru.yandex.webmaster3.storage.checklist.service;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.events2.HostEvent;
import ru.yandex.webmaster3.core.events2.HostEventId;
import ru.yandex.webmaster3.core.events2.client.HostEventLogClient;
import ru.yandex.webmaster3.core.events2.events.recheck.SiteProblemRecheckRequested;
import ru.yandex.webmaster3.storage.checklist.dao.SiteProblemsRecheckYDao;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.data.SiteProblemRecheckInfo;

/**
 * ishalaru
 * 16.04.2021
 **/
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SiteProblemsRecheckService {
    private final SiteProblemStorageService siteProblemStorageService;
    private final SiteProblemsRecheckYDao siteProblemsRecheckYDao;
    private Map<SiteProblemTypeEnum, ProblemRechecker> recheckers;
    private final HostEventLogClient hostEventLogClient;
    private final List<SiteProblemsProvider> availableProviders;
    private Map<SiteProblemTypeEnum, SiteProblemsProvider> problemsProvidersByType;

    public void init() {
        problemsProvidersByType = new EnumMap<>(SiteProblemTypeEnum.class);
        for (SiteProblemsProvider provider : availableProviders) {
            provider.getSupportedProblemTypes().forEach(type -> problemsProvidersByType.put(type, provider));
        }
    }

    public void requestSiteProblemRecheck(HostEventId eventId, WebmasterHostId hostId, SiteProblemTypeEnum problemType) {
        var problemHostId = siteProblemStorageService.toProblemHostId(hostId, problemType);
        RecheckInfo recheckInfo = getRecheckInfo(problemHostId, problemType);
        if (!recheckInfo.recheckable || recheckInfo.isInProgress) {
            log.info("Will not trigger recheck for host {} problem {}", problemHostId, problemType);
            return;
        }

        ProblemRechecker rechecker = recheckers.get(problemType);
        if (rechecker != null) {
            log.info("Will trigger recheck for host {} problem {}", problemHostId, problemType);
            rechecker.recheck(problemHostId, problemType);
        }

        siteProblemsRecheckYDao.updateRecheckState(problemHostId, problemType, true);
        hostEventLogClient.log(HostEvent.create(eventId, problemHostId, new SiteProblemRecheckRequested(problemType)));
    }

    private RecheckInfo getRecheckInfo(WebmasterHostId hostId, SiteProblemTypeEnum problemType) {
        AbstractProblemInfo problemInfo = problemsProvidersByType.get(problemType).getProblem(hostId, null, problemType);
        SiteProblemRecheckInfo recheckInfo = siteProblemsRecheckYDao.getRecheckInfo(hostId, problemType);

        boolean isInProgress = false;
        if (recheckInfo != null) {
            isInProgress = recheckInfo.getState(problemInfo == null ? null : problemInfo.getLastUpdate()).isInProgres;
        }

        SiteProblemState state = problemInfo == null ? problemType.getDefaultState() : problemInfo.getState();
        boolean recheckable = problemType.isRecheckable(state, problemInfo == null ? null : problemInfo.getContent());

        return new RecheckInfo(recheckable, isInProgress);
    }

    private class RecheckInfo {
        private final boolean recheckable;
        private final boolean isInProgress;

        public RecheckInfo(boolean recheckable, boolean isInProgress) {
            this.recheckable = recheckable;
            this.isInProgress = isInProgress;
        }
    }

    public void setRecheckers(Map<SiteProblemTypeEnum, ProblemRechecker> recheckers) {
        this.recheckers = recheckers;
    }

}
