package ru.yandex.webmaster3.storage.clickhouse;

import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

/**
 * @author aherman
 */
public class ClickhouseTableInfo implements ClickhouseTable, Comparable<ClickhouseTableInfo> {
    private final TableType type;
    private final UUID tableId;
    private final TableState state;
    private final DateTime updateDate;
    private final TableSource source;
    private final String sourcePath;
    private final String clickhouseFullName;
    private final String clickhouseDistributedFullName;
    private final String clickhouseLocalFullName;
    private final int shardCount;
    private final int partCount;

    public ClickhouseTableInfo(TableType type,
                               UUID tableId,
                               TableState state,
                               DateTime updateDate,
                               TableSource source,
                               String sourcePath,
                               String clickhouseFullName,
                               String clickhouseDistributedFullName,
                               String clickhouseLocalFullName,
                               Integer shardCount,
                               Integer partCount)
    {
        this.type = type;
        this.tableId = tableId;
        this.clickhouseFullName = clickhouseFullName;
        this.state = state;
        this.updateDate = updateDate;
        this.source = source;
        this.sourcePath = sourcePath;
        this.clickhouseDistributedFullName = clickhouseDistributedFullName;
        this.clickhouseLocalFullName = clickhouseLocalFullName;
        this.shardCount = shardCount == null ? 0 : shardCount;
        this.partCount = partCount == null ? 0 : partCount;
    }

    public TableType getType() {
        return type;
    }

    public UUID getTableId() {
        return tableId;
    }

    public String getClickhouseFullName() {
        return clickhouseFullName;
    }

    public TableState getState() {
        return state;
    }

    public DateTime getUpdateDate() {
        return updateDate;
    }

    public TableSource getSource() {
        return source;
    }

    public String getSourcePath() {
        return sourcePath;
    }

    public String toString() {
        return type + ":" + tableId + ":" + state;
    }

    @Override
    public String getDistributedTableName() {
        return clickhouseDistributedFullName;
    }

    @Override
    public String getLocalTableName() {
        return clickhouseLocalFullName;
    }

    @Override
    public int getShardCount() {
        return shardCount;
    }

    @Override
    public int getPartCount() {
        return partCount;
    }

    public ClickhouseTableInfo withState(TableState newState) {
        return new ClickhouseTableInfo(type, tableId, newState, DateTime.now(), source, sourcePath, clickhouseFullName,
                clickhouseDistributedFullName, clickhouseLocalFullName, shardCount, partCount);
    }

    public boolean isNewerThan(ClickhouseTableInfo other) {
        return UUIDs.unixTimestamp(this.tableId) > UUIDs.unixTimestamp(other.getTableId());
    }

    public long getUnixTimestamp() {
        return UUIDs.unixTimestamp(this.tableId);
    }

    @Override
    public int compareTo(@NotNull ClickhouseTableInfo other) {
        return Long.compare(getUnixTimestamp(), other.getUnixTimestamp());
    }
}
