package ru.yandex.webmaster3.storage.clickhouse;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.clickhouse.dao.LegacyClickhouseTablesYDao;
import ru.yandex.webmaster3.storage.clickhouse.table.ExternalDeletedLinkSamplesTable;
import ru.yandex.webmaster3.storage.clickhouse.table.ExternalLinkSamplesTable;
import ru.yandex.webmaster3.storage.clickhouse.table.InternalLinkSamplesTable;

/**
 * @author iceflame
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class LegacyMdbTableStorage implements TableProvider {
    private static final int CACHE_EXPIRATION_DURATION = 60; // seconds
    // минимальное время, которое должно пройти после добавления таблицы, чтобы клиенты начали ее использовать
    private static final int MINIMUM_TABLE_AGE = 60 * 1000;

    private volatile TablesState currentTablesState;
    private volatile TablesState preparedTablesState;

    private static final Object KEY = new Object();
    // тупой кэш, хранящий сразу все карту таблиц
    private final LoadingCache<Object, Map<TableType, List<ClickhouseTableInfo>>> onlineTablesCache =
            CacheBuilder.newBuilder()
                    .refreshAfterWrite(CACHE_EXPIRATION_DURATION, TimeUnit.SECONDS)
                    .build(new TableLoader());

    private class TableLoader extends CacheLoader<Object, Map<TableType, List<ClickhouseTableInfo>>> {

        @Override
        public Map<TableType, List<ClickhouseTableInfo>> load(Object key) throws Exception {
            // грузим все таблички. оставляем только онлайн и пре-онлайн
            // группируем по типу
            Map<TableType, List<ClickhouseTableInfo>> result = loadLatestTables().stream()
                    .filter(tableInfo -> tableInfo.getState().isOnline())
                    .sorted(Comparator.comparing(ClickhouseTableInfo::getUpdateDate))
                    .collect(Collectors.groupingBy(ClickhouseTableInfo::getType));
            return result;
        }
    }

    private final LegacyClickhouseTablesYDao clickhouseTablesYDao;

    public void init() {
        List<ClickhouseTableInfo> tables = loadLatestTables();
        tables = promoteToOnline(tables);
        // Ignore missing tables - better start with partial service than with nothing at all
        hasMissingTables(tables);
        tables.sort(getTableComparator());
        currentTablesState = new TablesState(tables);
        log.info("Start clickhouse tables storage with tables: {}", currentTablesState.getStateHash());
        logTables(currentTablesState);
    }

    private List<ClickhouseTableInfo> promoteToOnline(List<ClickhouseTableInfo> tables) {
        Map<TableType, ClickhouseTableInfo> onlineTables = new HashMap<>();

        for (ClickhouseTableInfo table : tables) {
            ClickhouseTableInfo otherTable = onlineTables.get(table.getType());
            if (table.getState() == TableState.ON_LINE) {
                if (otherTable == null || table.isNewerThan(otherTable)) {
                    onlineTables.put(table.getType(), table);
                }

            } else if (table.getState() == TableState.PRE_ONLINE) {
                if (otherTable == null || table.isNewerThan(otherTable)) {
                    log.info("Promote {} to {}", table, TableState.ON_LINE);
                    table = table.withState(TableState.ON_LINE);
                    onlineTables.put(table.getType(), table);
                }
            }
        }
        return new ArrayList<>(onlineTables.values());
    }

    private boolean hasMissingTables(List<ClickhouseTableInfo> tables) {
        Set<TableType> types = tables.stream().map(ClickhouseTableInfo::getType).collect(Collectors.toSet());
        Set<TableType> allType = Sets.newHashSet(TableType.values());
        Sets.SetView<TableType> missingTables = Sets.difference(allType, types);
        if (!missingTables.isEmpty()) {
            log.error("Missing tables: {}", missingTables);
            return true;
        }
        return false;
    }

    private void logTables(TablesState tablesState) {
        for (ClickhouseTableInfo table : tablesState.allTables) {
            log.info("Table: {} {} {}", table, table.getUpdateDate(), table.getClickhouseFullName());
        }
    }

    private ClickhouseTableInfo getTable(TableType type, TableState state) {
        try {
            List<ClickhouseTableInfo> tables = onlineTablesCache.get(KEY).getOrDefault(type, Collections.emptyList());
            // ищем первую таблицу с конца, у которой "возраст" достаточен
            // это необходимо для синхронизации кэшей на разных бэкэндах
            long millis = Instant.now().getMillis();
            int index = tables.size() - 1;
            while (index >= 0 && (millis - tables.get(index).getUpdateDate().getMillis() < MINIMUM_TABLE_AGE)) {
                index--;
            }
            if (index < 0) {
                throw new WebmasterException("Table not found: " + type + "=" + state,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(),
                                "Table not found: " + type + "=" + state));
            }
            return tables.get(index);
        } catch (ExecutionException e) {
            throw new WebmasterException("Error loading value from cache: " + type + "=" + state,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(),
                            "Table not found: " + type + "=" + state));
        }
    }

    public Pair<Boolean, TablesState> prepareTables() {
        List<ClickhouseTableInfo> nextTables = loadLatestTables();
        nextTables = promoteToOnline(nextTables);
        if (hasMissingTables(nextTables)) {
            return Pair.of(false, null);
        }
        TablesState nextState = new TablesState(nextTables);
        log.info("Prepare table state: {}", nextState.getStateHash());
        nextTables.sort(getTableComparator());
        logTables(nextState);
        this.preparedTablesState = nextState;
        return Pair.of(true, nextState);
    }

    private List<ClickhouseTableInfo> loadLatestTables() {
        return clickhouseTablesYDao.listTables();
    }

    public InternalLinkSamplesTable getInternalLinkSamplesTable() {
        ClickhouseTableInfo table = getTable(TableType.INTERNAL_LINK_SAMPLES, TableState.ON_LINE);
        return new InternalLinkSamplesTable(table);
    }

    public ExternalLinkSamplesTable getExternalLinkSamplesTable() {
        ClickhouseTableInfo table = getTable(TableType.EXTERNAL_LINK_SAMPLES, TableState.ON_LINE);
        return new ExternalLinkSamplesTable(table);
    }

    public ExternalDeletedLinkSamplesTable getExternalDeletedLinkSamplesTable() {
        ClickhouseTableInfo table = getTable(TableType.EXTERNAL_DELETED_LINK_SAMPLES, TableState.ON_LINE);
        return new ExternalDeletedLinkSamplesTable(table);
    }

    /**
     * Возвращает инфу о таблице нужного типа
     *
     * @param type
     * @return
     */
    public ClickhouseTableInfo getTable(TableType type) {
        return getTable(type, TableState.ON_LINE);
    }

    @NotNull
    private static Comparator<ClickhouseTableInfo> getTableComparator() {
        return Comparator.comparing(ClickhouseTableInfo::getType).thenComparingLong(t -> UUIDs.unixTimestamp(t.getTableId()));
    }

    public static class TablesState {
        private final List<ClickhouseTableInfo> allTables;

        public TablesState(List<ClickhouseTableInfo> allTables) {
            this.allTables = allTables;
        }

        public Optional<ClickhouseTableInfo> getTable(TableType type, TableState state) {
            return allTables.stream()
                    .filter(t -> t.getType() == type && t.getState() == state)
                    .sorted((t1, t2) -> t2.getUpdateDate().compareTo(t1.getUpdateDate()))
                    .findFirst();
        }

        public List<ClickhouseTableInfo> getAllTables() {
            return allTables;
        }

        public long getStateHash() {
            Hasher sip24Hasher = Hashing.sipHash24().newHasher();
            for (ClickhouseTableInfo table : allTables) {
                sip24Hasher.putString(table.getType().name(), StandardCharsets.UTF_8);
                sip24Hasher.putLong(table.getTableId().getLeastSignificantBits());
                sip24Hasher.putLong(table.getTableId().getMostSignificantBits());
                sip24Hasher.putInt(table.getState().value());
                sip24Hasher.putString(table.getClickhouseFullName(), StandardCharsets.UTF_8);
            }
            return sip24Hasher.hash().asLong();
        }
    }
}
