package ru.yandex.webmaster3.storage.clickhouse;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Suppliers;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.importer.dao.ClickhouseTablesYDao;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableInfo;
import ru.yandex.webmaster3.storage.importer.model.MdbClickhouseTableState;

/**
 * Created by Oleg Bazdyrev on 22/01/2021.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class MdbTableStorage {

    private static final long MINIMUM_TABLE_AGE = 60 * 1000L;
    private final ClickhouseTablesYDao clickhouseTablesYDao;

    private final Supplier<Map<String, List<MdbClickhouseTableInfo>>> onlineTablesSupplier = Suppliers.memoizeWithExpiration(
            this::loadOnlineTables, 1L, TimeUnit.MINUTES
    );

    private Map<String, List<MdbClickhouseTableInfo>> loadOnlineTables() {
        return clickhouseTablesYDao.listTables().stream()
                .filter(tableInfo -> tableInfo.getState() == MdbClickhouseTableState.ONLINE)
                .sorted(MdbClickhouseTableInfo.BY_UPDATED)
                .collect(Collectors.groupingBy(MdbClickhouseTableInfo::getId));
    }

    @Nullable
    public MdbClickhouseTableInfo getTable(String id) {
        List<MdbClickhouseTableInfo> tables = onlineTablesSupplier.get().getOrDefault(id, Collections.emptyList());
        // ищем первую таблицу с конца, у которой "возраст" достаточен
        // это необходимо для синхронизации кэшей на разных бэкэндах
        long millis = System.currentTimeMillis();
        int index = tables.size() - 1;
        while (index >= 0 && (millis - tables.get(index).getUpdated().getMillis() < MINIMUM_TABLE_AGE)) {
            index--;
        }
        if (index < 0) {
            throw new WebmasterException("Table " + id + " not found:",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), "Table " + id + " not found:"));
        }
        return tables.get(index);
    }

}
