package ru.yandex.webmaster3.storage.crawl.service;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.crawl.BaseCrawlInfo;
import ru.yandex.webmaster3.storage.crawl.RotorSettings;
import ru.yandex.webmaster3.storage.crawl.UserCrawlSettings;
import ru.yandex.webmaster3.storage.crawl.dao.BaseCrawlInfoYDao;
import ru.yandex.webmaster3.storage.crawl.dao.RotorSettingsYDao;
import ru.yandex.webmaster3.storage.crawl.dao.UserCrawlSettingsYDao;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

/**
 * @author avhaliullin
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class CrawlSettingsService {
    public static final UserCrawlSettings DEFAULT_USER_CRAWL_SETTINGS = new UserCrawlSettings(true, 1.0);

    private static final String ASTERISK = "*";
    private static final String DOT = ".";

    private final UserCrawlSettingsYDao userCrawlSettingsYDao;
    private final RotorSettingsYDao rotorSettingsYDao;
    private final BaseCrawlInfoYDao baseCrawlInfoYDao;
    private final SettingsService settingsService;

    public BaseCrawlInfo getBaseCrawlInfo(WebmasterHostId hostId) {
        DateTime addDate = DateTime.parse(settingsService.getSettingCached(CommonDataType.CRAWL_DELAY_IMPORT_DATE).getValue());
        String punycodeHostname = hostId.getPunycodeHostname();
        BaseCrawlInfo result;
        try {
            result = baseCrawlInfoYDao.getInfo(hostId.getSchema().getSchemaPrefix() + punycodeHostname, addDate);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get base crawl settings",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
        if (result != null) {
            return result;
        }
        String[] levels = punycodeHostname.split("\\.");
        for (int startLevelIndex = 0; startLevelIndex <= levels.length; startLevelIndex++) {
            String hostPattern = StringUtils.join(levels, DOT, startLevelIndex, levels.length);
            if (hostPattern.isEmpty()) {
                hostPattern = ASTERISK;
            } else {
                hostPattern = ASTERISK + DOT + hostPattern;
            }

            BaseCrawlInfo info;
            try {
                info = baseCrawlInfoYDao.getInfo(hostId.getSchema().getSchemaPrefix() + hostPattern, addDate);
            } catch (WebmasterYdbException e) {
                throw new WebmasterException("Failed to get base crawl settings",
                        new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
            }
            if (info != null) {
                result = info;
                break;
            }
        }
        if (result == null) {
            result = new BaseCrawlInfo(
                    Double.valueOf(settingsService.getSettingCached(CommonDataType.CRAWL_DELAY_DEFAULT_RPS).getValue()));
        }
        return result;
    }

    public UserCrawlSettings getUserCrawlSettings(WebmasterHostId hostId) {
        try {
            var result = userCrawlSettingsYDao.getSettings(hostId);
            if (result == null) {
                result = DEFAULT_USER_CRAWL_SETTINGS;
            }

            return result;
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get user crawl settings",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public RotorSettings getRotorSettings(WebmasterHostId hostId) {
        try {
            var result = rotorSettingsYDao.getSettings(hostId);
            if (result == null) {
                result = new RotorSettings(hostId, RotorSettings.RenderSettings.RENDER_AUTO);
            }

            return result;
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get user crawl settings",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public void updateUserCrawlSettings(WebmasterHostId hostId, UserCrawlSettings settings) {
        userCrawlSettingsYDao.updateSettings(hostId, settings);
    }

    public void updateRotorSettings(RotorSettings settings) {
        rotorSettingsYDao.updateSettings(settings);
    }
}
