package ru.yandex.webmaster3.storage.delegation;

import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author avhaliullin
 */
@Repository
public class UserHostDelegationsYDao extends AbstractYDao {
    public UserHostDelegationsYDao() {
        super(PREFIX_DELEGATION, "user_host_delegations");
    }

    public void addDelegationRecord(HostDelegationRecord delegationRecord) {
        upsert(
                F.HOST_ID.value(delegationRecord.getHostId()),
                F.FROM_USER_ID.value(delegationRecord.getFromUser()),
                F.TO_USER_ID.value(delegationRecord.getToUser()),
                F.DELEGATED_DATE.value(delegationRecord.getDelegatedDate()),
                F.CANCELLED.value(delegationRecord.isCancelled()),
                F.CANCELLED_BY_USER_ID.value(delegationRecord.getCancelledByUser()),
                F.CANCELLED_DATE.value(delegationRecord.getCancelledDate())
        ).execute();
    }

    private static final ValueDataMapper<HostDelegationRecord> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, HostDelegationRecord::getHostId),
            Pair.of(F.TO_USER_ID, HostDelegationRecord::getToUser),
            Pair.of(F.FROM_USER_ID, HostDelegationRecord::getFromUser),
            Pair.of(F.DELEGATED_DATE, HostDelegationRecord::getDelegatedDate),
            Pair.of(F.CANCELLED, HostDelegationRecord::isCancelled),
            Pair.of(F.CANCELLED_DATE, HostDelegationRecord::getCancelledDate),
            Pair.of(F.CANCELLED_BY_USER_ID, HostDelegationRecord::getCancelledByUser)
    );

    public List<HostDelegationRecord> getDelegationsForHost(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .queryForList();
    }

    public HostDelegationRecord getDelegationForHostToUser(WebmasterHostId hostId, long toUserId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .and(F.TO_USER_ID.eq(toUserId))
                .queryOne();
    }

    public void forEach(Consumer<HostDelegationRecord> consumer) {
        streamReader(MAPPER, consumer);
    }

    public void deleteForFromUser(WebmasterUser user) {
        var sel = select(PK_MAPPER)
                .secondaryIndex("from_user_id_index")
                .where(F.FROM_USER_ID.eq(user.getUserId()))
                .getStatement();
        delete().on(sel).execute();
    }

    public void deleteForToUser(WebmasterUser user) {
        var sel = select(PK_MAPPER)
                .secondaryIndex("to_user_id_index")
                .where(F.TO_USER_ID.eq(user.getUserId()))
                .getStatement();
        delete().on(sel).execute();
    }

    private static final DataMapper<HostDelegationRecord> MAPPER = DataMapper.create(
            F.FROM_USER_ID, F.TO_USER_ID, F.HOST_ID, F.DELEGATED_DATE, F.CANCELLED, F.CANCELLED_DATE, F.CANCELLED_BY_USER_ID,
            HostDelegationRecord::new
    );

    private static final DataMapper<Pair<WebmasterHostId, Long>> PK_MAPPER = DataMapper.create(F.HOST_ID, F.TO_USER_ID, Pair::of);

    public void addDelegationRecord(Collection<HostDelegationRecord> delegationRecords) {
        batchUpdate(VALUE_MAPPER, delegationRecords).execute();
    }

    private static class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<Long> TO_USER_ID = Fields.longField("to_user_id");
        static final Field<Long> FROM_USER_ID = Fields.longField("from_user_id");
        static final Field<DateTime> DELEGATED_DATE = Fields.jodaDateTimeField("delegated_date");
        static final Field<Boolean> CANCELLED = Fields.boolField("cancelled").withDefault(false);
        static final Field<DateTime> CANCELLED_DATE = Fields.jodaDateTimeField("cancelled_date").makeOptional();
        static final Field<Long> CANCELLED_BY_USER_ID = Fields.longField("cancelled_by_user_id").makeOptional();
    }
}
