package ru.yandex.webmaster3.storage.delurl.dao;

import java.util.*;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.delurl.DelUrlRequest;
import ru.yandex.webmaster3.core.delurl.DelurlState;
import ru.yandex.webmaster3.core.delurl.DelurlType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.QueryBuilder;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author leonidrom
 */
@Service
public class DelUrlRequestsYDao extends AbstractYDao {
    private static final String TABLE_NAME = "delurl_requests";
    private static final String ADD_DATE_INDEX = "add_date_index";

    @Autowired
    public DelUrlRequestsYDao() {
        super("/webmaster3/delurl", TABLE_NAME);
    }

    public void addBatch(List<DelUrlRequest> batch) {
        if (batch.isEmpty()) {
            return;
        }

        execute(batchUpdate(INSERT_VALUE_MAPPER, batch));
    }

    public void updateBatch(List<DelUrlRequest> batch) {
        if (batch.isEmpty()) {
            return;
        }

        execute(batchUpdate(UPDATE_VALUE_MAPPER, batch));
    }

    @Nullable
    public DelUrlRequest get(WebmasterHostId hostId, UUID urlId) {
        var sel = select(MAPPER).where(F.HOST_ID.eq(hostId)).and(F.URL_ID.eq(urlId));
        return queryOne(sel.getStatement(), MAPPER);
    }

    public int count(WebmasterHostId hostId, DateTime fromDate, DateTime toDate) {
        var st = select().countAll().where(F.HOST_ID.eq(hostId));
        if (fromDate != null) {
            st.and(F.ADD_DATE.gte(fromDate));
        }

        if (toDate != null) {
            st.and(F.ADD_DATE.lte(toDate));
        }

        return queryOne(st.getStatement(), DataMapper.SINGLE_COLUMN_INT_MAPPER);
    }

    public List<DelUrlRequest> list(WebmasterHostId hostId, DateTime fromDate, DateTime toDate, int skip, int limit) {
        var st = select(MAPPER).where(F.HOST_ID.eq(hostId));
        if (fromDate != null) {
            st.and(F.ADD_DATE.gte(fromDate));
        }

        if (toDate != null) {
            st.and(F.ADD_DATE.lte(toDate));
        }

        st.order(F.ADD_DATE.desc());
        st.limit(skip, limit);

        // используется для пагинации с маленьким размером страницы, так что в лимит YDB не упираемся
        return queryForList(st.getStatement(), MAPPER);
    }

    public List<DelUrlRequest> list(DateTime fromDate, DateTime toDate) {
        var st = select(MAPPER).secondaryIndex(ADD_DATE_INDEX)
                .where(F.ADD_DATE.gte(fromDate))
                .and(F.ADD_DATE.lte(toDate))
                // сортируем по первичному ключу
                .order(F.HOST_ID.asc())
                .order(F.URL_ID.asc());

        return queryForList(st, MAPPER, list -> {
            DelUrlRequest maxReq = list.get(list.size() - 1);
            var cond = QueryBuilder.or(
                    QueryBuilder.and(F.HOST_ID.eq(maxReq.getHostId()), F.URL_ID.gt(maxReq.getRequestId())),
                    F.HOST_ID.gt(maxReq.getHostId())
            );

            return st.cont(cond).getStatement();
        });
    }

    public List<DelUrlRequest> list(WebmasterHostId hostId, DateTime fromDate) {
        var st = select(MAPPER).where(F.HOST_ID.eq(hostId));
        if (fromDate != null) {
            st.and(F.ADD_DATE.gte(fromDate));
        }

        // сортируем по первичному ключу
        st.order(F.URL_ID.asc());

        return queryForList(st.getStatement(), MAPPER, list -> {
            DelUrlRequest maxReq = list.get(list.size() - 1);
            return st.cont(F.URL_ID.gt(maxReq.getRequestId())).getStatement();
        });
    }

    public List<String> listRequestIds(WebmasterHostId hostId, DateTime fromDate) {
        var st = select(REQUEST_ID_MAPPER).where(F.HOST_ID.eq(hostId));
        if (fromDate != null) {
            st.and(F.ADD_DATE.gte(fromDate));
        }

        // используется для дедупликации возможных недавних ретраев балансера, поэтому лимит YDB нас здесь не сломает
        return queryForList(st.getStatement(), REQUEST_ID_MAPPER);
    }

    public void deleteForUser(long userId) {
        var sel = select(PK_MAPPER).secondaryIndex(USER_ID_INDEX).where(F.USER_UID.eq(userId)).getStatement();
        delete().on(sel).execute();
    }

    private static final DataMapper<String> REQUEST_ID_MAPPER = DataMapper.create(F.REQUEST_ID, r -> r);

    private static final DataMapper<Pair<WebmasterHostId, UUID>> PK_MAPPER = DataMapper.create(F.HOST_ID, F.URL_ID, Pair::of);

    private static final DataMapper<DelUrlRequest> MAPPER = DataMapper.create(
            F.URL_ID, F.HOST_ID, F.RELATIVE_URL, F.STATE, F.TYPE, F.ADD_DATE, F.UPDATE_DATE,
            F.ALLOWED_IN_ROBOTS_TXT, F.HTTP_CODE, F.NOINDEX, F.USER_UID, F.REQUEST_ID, F.RETRIES_COUNT, F.ERROR_MESSAGE,
            ((urlId, hostId, relativeUrl, state, type, addDate, updateDate, isAllowedInRobots, httpCode, isNoIndex,  userId, requestId, retriesCount, errorMessage) ->
                    new DelUrlRequest(
                            urlId, hostId, relativeUrl, state, type, addDate, updateDate,
                            isAllowedInRobots, httpCode, isNoIndex, userId, requestId, retriesCount, errorMessage).withStaleCheck())
    );

    private static final ValueDataMapper<DelUrlRequest> INSERT_VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.HOST_ID, r -> F.HOST_ID.get(r.getHostId())),
            Pair.of(F.URL_ID, r -> F.URL_ID.get(r.getRequestId())),
            Pair.of(F.RELATIVE_URL, r -> F.RELATIVE_URL.get(r.getRelativeUrl())),
            Pair.of(F.STATE, r -> F.STATE.get(r.getState())),
            Pair.of(F.TYPE, r -> F.TYPE.get(r.getType())),
            Pair.of(F.ADD_DATE, r -> F.ADD_DATE.get(r.getAddDate())),
            Pair.of(F.UPDATE_DATE, r -> F.UPDATE_DATE.get(r.getUpdateDate())),
            Pair.of(F.ALLOWED_IN_ROBOTS_TXT, r -> F.ALLOWED_IN_ROBOTS_TXT.get(r.isAllowedInRobotsTxt())),
            Pair.of(F.HTTP_CODE, r -> F.HTTP_CODE.get(r.getHttpCode())),
            Pair.of(F.NOINDEX, r -> F.NOINDEX.get(r.isNoindex())),
            Pair.of(F.USER_UID, r -> F.USER_UID.get(r.getUserUid())),
            Pair.of(F.REQUEST_ID, r -> F.REQUEST_ID.get(r.getBalancerRequestId())),
            Pair.of(F.RETRIES_COUNT, r -> F.RETRIES_COUNT.get(r.getRetriesCount())),
            Pair.of(F.ERROR_MESSAGE, r -> F.ERROR_MESSAGE.get(r.getErrorMessage()))
    );

    private static final ValueDataMapper<DelUrlRequest> UPDATE_VALUE_MAPPER = ValueDataMapper.create(
            Pair.of(F.HOST_ID, r -> F.HOST_ID.get(r.getHostId())),
            Pair.of(F.URL_ID, r -> F.URL_ID.get(r.getRequestId())),
            Pair.of(F.STATE, r -> F.STATE.get(r.getState())),
            Pair.of(F.UPDATE_DATE, r -> F.UPDATE_DATE.get(r.getUpdateDate())),
            Pair.of(F.ALLOWED_IN_ROBOTS_TXT, r -> F.ALLOWED_IN_ROBOTS_TXT.get(r.isAllowedInRobotsTxt())),
            Pair.of(F.HTTP_CODE, r -> F.HTTP_CODE.get(r.getHttpCode())),
            Pair.of(F.NOINDEX, r -> F.NOINDEX.get(r.isNoindex())),
            Pair.of(F.RETRIES_COUNT, r -> F.RETRIES_COUNT.get(r.getRetriesCount())),
            Pair.of(F.ERROR_MESSAGE, r -> F.ERROR_MESSAGE.get(r.getErrorMessage()))
    );

    private static class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<UUID> URL_ID = Fields.uuidField("url_id");
        static final Field<String> RELATIVE_URL = Fields.stringField("relative_url");
        static final Field<DelurlState> STATE = Fields.intEnumField("state", DelurlState.R);
        static final Field<DelurlType> TYPE = Fields.intEnumField("type", DelurlType.R).withDefault(DelurlType.URL);
        static final Field<DateTime> ADD_DATE = Fields.jodaDateTimeField("add_date");
        static final Field<DateTime> UPDATE_DATE = Fields.jodaDateTimeField("update_date");
        static final Field<Boolean> ALLOWED_IN_ROBOTS_TXT = Fields.boolField("allowed_in_robots_txt").withDefault(true);
        static final Field<Integer> HTTP_CODE = Fields.intField("http_code");
        static final Field<Boolean> NOINDEX = Fields.boolField("noindex");
        static final Field<Long> USER_UID = Fields.longField("user_uid");
        static final Field<String> REQUEST_ID = Fields.stringField("request_id");
        static final Field<Integer> RETRIES_COUNT = Fields.intField("retries_count").withDefault(0);
        static final Field<String> ERROR_MESSAGE = Fields.stringField("error_message").makeOptional();
    }
}
