package ru.yandex.webmaster3.storage.digest;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import NWebmaster.proto.digest.Digest;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.clickhouse.system.data.ClickhouseSystemTableInfo;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.From;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * @author avhaliullin
 */
public class DigestMessagesCHDao extends AbstractClickhouseDao implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(DigestMessagesCHDao.class);
    private static final String DATABASE = DigestTableUtil.DATABASE;
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormat.forPattern("yyyyMMdd");

    @Setter
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;

    public int countMessages(LocalDate digestDate) throws ClickhouseException {
        Statement st = QueryBuilder.select("count()")
                .from(DATABASE, getTableName(digestDate));
        return getClickhouseServer().queryOne(
                ClickhouseQueryContext.useDefaults().setHost(pickHost(digestDate)),
                st.toString(),
                row -> row.getLongUnsafe(0)
        ).orElse(0L).intValue();
    }

    public List<DigestMessage> getMessagesForHost(LocalDate digestDate, WebmasterHostId hostId) throws ClickhouseException {
        Statement st = selectAllFields(digestDate)
                .where(QueryBuilder.eq(F.HOST_ID, hostId.toString()));
        return getClickhouseServer().queryAll(
                ClickhouseQueryContext.useDefaults().setHost(pickHost(digestDate)),
                st.toString(),
                getMapper(digestDate)
        );
    }

    public DigestMessage getMessagesForHost(LocalDate digestDate, long userId, WebmasterHostId hostId) throws ClickhouseException {
        Statement st = selectAllFields(digestDate)
                .where(QueryBuilder.eq(F.USER_ID, userId))
                .and(QueryBuilder.eq(F.HOST_ID, hostId.toString()));
        return getClickhouseServer().queryOne(
                ClickhouseQueryContext.useDefaults().setHost(pickHost(digestDate)),
                st.toString(),
                getMapper(digestDate)
        ).orElse(null);
    }

    public List<DigestMessage> getMessages(LocalDate digestDate, int fromOffset, int size) throws ClickhouseException {
        // прямой запрос сжирает слишком много памяти, поэтому делим запрос на два
        // во внутреннем собираем только ключи
        Statement innerQuery = QueryBuilder.select(F.USER_ID, F.HOST_ID)
                .from(DATABASE, getTableName(digestDate))
                .orderBy(Arrays.asList(Pair.of(F.USER_ID, OrderBy.Direction.ASC), Pair.of(F.HOST_ID, OrderBy.Direction.ASC)))
                .limit(fromOffset, size);

        Statement st = selectAllFields(digestDate)
                .where(QueryBuilder.inSubquery(innerQuery, F.USER_ID, F.HOST_ID));


        return getClickhouseServer().queryAll(
                ClickhouseQueryContext.useDefaults().setHost(pickHost(digestDate)),
                st.toString(),
                getMapper(digestDate)
        );
    }

    public Set<LocalDate> getTableDates() {
        return getClickhouseServer().getHosts().stream().filter(ClickhouseHost::isUp)
                .flatMap(host -> clickhouseSystemTablesCHDao.getTablesForPrefix(host, DATABASE, "digest_notification").stream()
                        .map(ClickhouseSystemTableInfo::getName))
                .filter(tableName -> tableName.startsWith("digest_notification_"))
                .map(tableName -> LocalDate.parse(tableName.substring(20), TIME_FORMATTER))
                .collect(Collectors.toSet());
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        long userId = user.getUserId();
        Collection<LocalDate> tableDates = getTableDates();
        for (var tableDate : tableDates) {
            String tableName = DigestTableUtil.tableName(tableDate);
            String query = String.format("ALTER TABLE %s DELETE WHERE user_id=%s", DATABASE + "." + tableName, userId);
            getClickhouseServer().execute(pickHost(tableDate), query);
        }
    }

    private static From selectAllFields(LocalDate digestDate) {
        return QueryBuilder.select(F.USER_ID, F.HOST_ID, F.EMAIL, F.LANG, F.NOTIFICATION_TYPE, F.DATA)
                .from(DATABASE, getTableName(digestDate));
    }

    private ClickhouseHost pickHost(LocalDate date) {
        String tableName = DigestTableUtil.tableName(date);
        Where countTablesQuery = QueryBuilder.select()
                .countAll()
                .from("system.tables")
                .where(QueryBuilder.eq("database", DATABASE))
                .and(QueryBuilder.eq("name", tableName));

        ClickhouseHost targetHost = null;
        List<ClickhouseHost> hosts = new ArrayList<>(getClickhouseServer().getHosts());
        // чтобы все сервера мучались равномерно
        Collections.shuffle(hosts);
        for (ClickhouseHost host : ClickhouseServer.alive().apply(hosts)) {
            try {
                long count = getClickhouseServer().executeWithFixedHost(host, () -> queryOne(countTablesQuery,
                        r -> r.getLongUnsafe(0)).orElse(0L));
                if (count > 0) {
                    targetHost = host;
                    break;
                }
            } catch (ClickhouseException e) {
                log.error("Failed to show clickhouse tables on " + host, e);
            }
        }
        if (targetHost == null) {
            throw new WebmasterException("Get messages failed",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(getClass(), null),
                    new ClickhouseException("No clickhouse hosts avaliable with table " + tableName, countTablesQuery.toString(), null));
        }
        return targetHost;
    }

    private static Function<CHRow, DigestMessage> getMapper(LocalDate digestPeriodEnd) {
        return row -> {
            try {
                return new DigestMessage(
                        row.getLong(F.USER_ID),
                        row.getHostId(F.HOST_ID),
                        row.getString(F.EMAIL),
                        LanguageEnum.fromString(row.getString(F.LANG)),
                        NotificationType.valueOf(row.getString(F.NOTIFICATION_TYPE)),
                        Digest.DigestWeeklyReportMessage.parseFrom(row.getBytes(F.DATA)),
                        digestPeriodEnd
                );
            } catch (InvalidProtocolBufferException e) {
                throw new RuntimeException(e);
            }
        };
    }

    private static class F extends DigestTableUtil.F {}

    private static String getTableName(LocalDate digestDate) {
        return DigestTableUtil.tableName(digestDate);
    }
}
