package ru.yandex.webmaster3.storage.download;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Preconditions;
import org.apache.http.Header;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;

/**
 * Created by ifilippov5 on 07.02.17.
 */
public class MDSService extends AbstractExternalAPIService {
    private static final Logger log = LoggerFactory.getLogger(MDSService.class);
    private static final int CONNECT_TIMEOUT = 500;
    private static final int SOCKET_TIMEOUT = (int) TimeUnit.SECONDS.toMillis(600);
    private static final Duration DEFAULT_TTL = Duration.standardHours(1L);

    private final String uploadHost;
    private final String readHost;
    private final String namespace;
    private final String authUpload;

    private CloseableHttpClient httpClient;

    @Autowired
    public MDSService(String uploadHost, String readHost, String namespace, String authUpload) {
        this.uploadHost = uploadHost;
        this.readHost = readHost;
        this.namespace = namespace;
        this.authUpload = authUpload;
    }

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setMaxConnPerRoute(20)
                .setMaxConnTotal(20)
                .setDefaultRequestConfig(requestConfig).build();
    }

    public String uploadFileAndGetDownloadLink(byte[] data, String fileName) {
        return uploadFileAndGetDownloadLink(new ByteArrayInputStream(data), fileName);
    }

    public String uploadFileAndGetDownloadLink(InputStream dataStream, String fileName) {
        return getDownloadLinkFromMds(uploadFileAndGetMDSKey(dataStream, fileName, DEFAULT_TTL, null));
    }

    @ExternalDependencyMethod("upload")
    public String uploadFileAndGetMDSKey(InputStream dataStream, String fileName, Duration ttl, String contentType) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            String path = "/upload-" + namespace + "/" + fileName;
            HttpUriRequest request;
            try {
                URIBuilder uriBuilder = new URIBuilder()
                        .setScheme("http")
                        .setHost(uploadHost)
                        .setPath(path)
                        .addParameter("expire", ttl.getStandardHours() + "h");

                HttpPost httpPost = new HttpPost(uriBuilder.build());
                if (log.isDebugEnabled()) {
                    log.debug("download POST on {} and post params: ", uriBuilder);
                }
                httpPost.setHeader("Authorization", authUpload);
                if (contentType != null) {
                    httpPost.setHeader("X-Mds-Content-Type", contentType);
                }
                httpPost.setEntity(new InputStreamEntity(dataStream));
                request = httpPost;
            } catch (URISyntaxException e) {
                throw new WebmasterException("Building upload request failed",
                        new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, e), e);
            }

            try {
                try (var httpResponse = httpClient.execute(request)) {
                    int responseStatus = httpResponse.getStatusLine().getStatusCode();
                    log.debug("Response status code: {}", responseStatus);
                    String responseStr = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
                    log.debug("Download response:\n{}", responseStr);

                    if (responseStatus == HttpStatus.SC_FORBIDDEN) {
                        log.warn("MDS objects overriding is not allowed");
                        return extractKeyOfExistingFile(responseStr);
                    } else if (responseStatus == HttpStatus.SC_BAD_REQUEST) {
                        log.warn("Bad response code, file is empty: {}", responseStatus);
                        throw new WebmasterException("400 Bad Request",
                                new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, null));
                    } else if (responseStatus != HttpStatus.SC_OK) {
                        log.warn("Bad response code: {}", responseStatus);
                        throw new WebmasterException("MDS request " + request + " responded with " + httpResponse.getStatusLine(),
                                new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, null));
                    }

                    return extractKey(responseStr);
                }
            } catch (Exception e) {
                log.error("Upload to MDS failed", e);
                throw new WebmasterException("Upload request failed",
                        new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, e), e);
            }
        });
    }

    public boolean deleteFile(DownloadInfo info) {
        // extract key
        String path = URI.create(info.getPublicUrlMds()).getPath();
        String prefix = "/get-" + namespace + "/";
        Preconditions.checkState(path.startsWith(prefix));
        String key = path.substring(prefix.length());
        return deleteFile(key);
    }

    @ExternalDependencyMethod("delete")
    public boolean deleteFile(String key) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            String path = "/delete-" + namespace + "/" + key;
            HttpGet request;
            try {
                URIBuilder uriBuilder = new URIBuilder()
                        .setScheme("http")
                        .setHost(uploadHost)
                        .setPath(path);

                request = new HttpGet(uriBuilder.build());
                if (log.isDebugEnabled()) {
                    log.debug("delete GET on {} and get params: ", uriBuilder);
                }
                request.setHeader("Authorization", authUpload);
            } catch (URISyntaxException e) {
                throw new WebmasterException("Building upload request failed",
                        new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, e), e);
            }

            try {
                try (var httpResponse = httpClient.execute(request)) {
                    int responseStatus = httpResponse.getStatusLine().getStatusCode();
                    log.debug("Response status code: {}", responseStatus);
                    String responseStr = EntityUtils.toString(httpResponse.getEntity(), StandardCharsets.UTF_8);
                    log.debug("Delete response:\n{}", responseStr);

                    if (responseStatus != HttpStatus.SC_OK) {
                        log.warn("Bad response code: {}", responseStatus);
                        throw new WebmasterException("MDS request " + request + " responded with " + httpResponse.getStatusLine(),
                                new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, null));
                    }
                }
                return true;
            } catch (IOException e) {
                log.error("Upload to MDS failed", e);
                throw new WebmasterException("Upload request failed",
                        new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, e), e);
            }
        });
    }

    public String getLinkFromMds(String fileKey) {
        return "https://" + readHost + "/get-" + namespace + "/" + fileKey;
    }

    public String getDownloadLinkFromMds(String fileKey) {
        String path = "/get-" + namespace + "/" + fileKey;
        try {
            var uriBuilder = new URIBuilder()
                    .setScheme("https")
                    .setHost(readHost)
                    .setPath(path)
                    .setParameter("disposition", "1");

            return uriBuilder.build().toString() + "&content_type=application/octet-stream";
        } catch (URISyntaxException e) {
            throw new WebmasterException("Build download uri failed",
                    new WebmasterErrorResponse.MdsErrorResponse(MDSService.class, e), e);
        }
    }

    // Если понадобится редирект на скачивание файла
    private String extractLocation(CloseableHttpResponse responseS) {
        Header[] headers = responseS.getAllHeaders();
        for (Header header : headers) {
            if (header.getName().equals("Location")) {
                return header.getValue();
            }
        }
        return null;
    }

    private String extractKey(String xmlData) {
        var stringReader = new StringReader(xmlData);
        var builder = new SAXBuilder();
        try {
            var eventDocument = builder.build(stringReader);
            var rootElement = eventDocument.getRootElement();
            return rootElement.getAttribute("key").getValue();
        } catch (IOException | JDOMException e) {
            throw new WebmasterException("MDS response parsing failed",
                    new WebmasterErrorResponse.UnableToParseXMlDataResponse(MDSService.class, e), e);
        }
    }

    private String extractKeyOfExistingFile(String xmlData) {
        var stringReader = new StringReader(xmlData);
        var builder = new SAXBuilder();
        try {
            var eventDocument = builder.build(stringReader);
            var rootElement = eventDocument.getRootElement();
            return rootElement.getChild("key").getValue();
        } catch (IOException | JDOMException e) {
            throw new WebmasterException("MDS response parsing failed",
                    new WebmasterErrorResponse.UnableToParseXMlDataResponse(MDSService.class, e), e);
        }
    }
}
