package ru.yandex.webmaster3.storage.download.searchquery;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.download.FileFormat;
import ru.yandex.webmaster3.core.download.searchquery.AbstractWriter;
import ru.yandex.webmaster3.core.searchquery.*;
import ru.yandex.webmaster3.storage.http.searchquery.statistic.StatisticsHelper;
import ru.yandex.webmaster3.storage.searchquery.*;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author lester
 */
@Service("urlsExportService")
public class UrlsExportService {
    private static final Logger log = LoggerFactory.getLogger(UrlsExportService.class);
    @Autowired
    protected UrlStatisticsService2 urlStatisticsService2;

    static final List<QueryIndicator> DEFAULT_INDICATORS = Arrays.stream(QueryIndicator.values())
            .filter(qi -> qi.isUsable(IndicatorUsage.IN_STATISTICS))
            .collect(Collectors.toList());

    public byte[] getAllUrlsAllIndicatorsStatistics(
            WebmasterHostId hostId, LocalDate userRangeStart, LocalDate userRangeEnd, Integer[] regionId,
            RegionInclusion regionInclusion, SpecialGroup specialGroup, DeviceType deviceType, QueryIndicator orderBy,
            OrderDirection orderDirection, FileFormat exportFormat) {
        if (orderBy == null) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        if (orderDirection == null) {
            orderDirection = OrderDirection.DESC;
        }

        List<QueryIndicator> indicators = DEFAULT_INDICATORS;
        final Set<Integer> regionIds = StatisticsHelper.asSet(10, regionId);

        Stopwatch sw = Stopwatch.createStarted();
        List<PathId> pathIds =
                urlStatisticsService2.getPathIds(hostId, specialGroup,
                        regionIds, regionInclusion,
                        userRangeStart, userRangeEnd,
                        Collections.emptyList(), deviceType,
                        orderBy, orderDirection, 0, 3000
                );
        sw.stop();
        log.info("Get urls: {}", sw.elapsed(TimeUnit.MILLISECONDS));

        RangeSet<LocalDate> rangeSet = RangeFactory.singleRange(userRangeStart, userRangeEnd);

        sw.start();
        Pair<Map<PathId, String>, List<PathStat>> statistics = urlStatisticsService2.getPathStatistics(
                hostId, specialGroup, indicators,
                regionInclusion, regionIds,
                pathIds,
                rangeSet.span().lowerEndpoint(), rangeSet.span().upperEndpoint(), deviceType);
        sw.stop();
        log.info("Get data: {}", sw.elapsed(TimeUnit.MILLISECONDS));

        List<Object> names = new ArrayList<>();
        names.add("Path");
        names.add("Dates range");
        for (QueryIndicator indicator : indicators) {
            names.add(QueriesExportService.translateIndicator(indicator));
        }

        AbstractWriter writer = exportFormat.getWriter();
        log.debug("Write {} to export file", names);
        writer.write(names);

        sw.start();
        Map<PathId, AccumulatorMap> accumulators = new HashMap<>();
        for (PathStat pathStat : statistics.getValue()) {
            AccumulatorMap ac = accumulators
                    .computeIfAbsent(pathStat.getPathId(), q -> AccumulatorMap.create(indicators, rangeSet));
            ac.apply(pathStat);
        }

        for (PathId id : pathIds) {
            String pathText = statistics.getKey().get(id);
            if (!StringUtils.isEmpty(pathText)) {
                List<Object> line = new ArrayList<>();
                line.add(pathText);
                line.add(QueriesExportService.rangeToText(rangeSet.span()));

                AccumulatorMap accumulatorMap = accumulators
                        .computeIfAbsent(id, q -> AccumulatorMap.create(indicators, rangeSet));

                for (QueryIndicator indicator : indicators) {
                    List<Pair<Range<LocalDate>, Double>> indicatorStat = accumulatorMap.getIndicator(indicator);
                    Pair<Range<LocalDate>, Double> pair = indicatorStat.get(0);
                    line.add(indicator.format(pair.getValue()));
                }
                writer.write(line);
            }
        }
        sw.stop();
        log.info("Create : {}", sw.elapsed(TimeUnit.MILLISECONDS));
        try {
            writer.close();
        } catch (IOException e) {
            log.error("Error with close writer", e);
        }
        return writer.getBytes();
    }

    public byte[] getAllUrlsIndicatorStatistics(
            WebmasterHostId hostId, LocalDate userRangeStart, LocalDate userRangeEnd, Integer[] regionId,
            RegionInclusion regionInclusion, SpecialGroup specialGroup, QueryIndicator queryIndicator, DeviceType deviceType,
            QueryIndicator orderBy, OrderDirection orderDirection, AggregatePeriod period, FileFormat exportFormat) {

        if (orderBy == null) {
            orderBy = QueryIndicator.TOTAL_SHOWS_COUNT;
        }
        if (orderDirection == null) {
            orderDirection = OrderDirection.DESC;
        }
        if (period == null) {
            period = AggregatePeriod.DAY;
        }

        final Set<Integer> regionIds = StatisticsHelper.asSet(10, regionId);
        final List<QueryIndicator> indicators = StatisticsHelper.asList(1, queryIndicator);

        List<PathId> pathIds =
                urlStatisticsService2.getPathIds(hostId, specialGroup,
                        regionIds, regionInclusion,
                        userRangeStart, userRangeEnd,
                        Collections.emptyList(), deviceType,
                        orderBy, orderDirection, 0, 3000
                );

        Map<PathId, String> idToPathText = urlStatisticsService2.getPathTexts(hostId, specialGroup, pathIds);

        RangeSet<LocalDate> userRangeSet = RangeFactory.singleRange(userRangeStart, userRangeEnd);
        RangeSet<LocalDate> intervalRangeSet = RangeFactory.createRanges(userRangeStart, userRangeEnd, period, false, false);

        Range<LocalDate> pathRange = userRangeSet.span().span(intervalRangeSet.span());

        List<PathStat> wholeStat = urlStatisticsService2.getStatistics(specialGroup, hostId,
                indicators, pathIds,
                regionIds, regionInclusion,
                pathRange.lowerEndpoint(), pathRange.upperEndpoint(), deviceType);

        Map<PathId, AccumulatorMap> accumulators = new HashMap<>();
        Function<PathId, AccumulatorMap> accumulatorFactory = q -> AccumulatorMap.create(indicators, intervalRangeSet);
        for (PathStat pathStat : wholeStat) {
            AccumulatorMap ac = accumulators.computeIfAbsent(pathStat.getPathId(), accumulatorFactory);
            ac.apply(pathStat);
        }

        List<Object> names = new ArrayList<>();
        names.add("Path");

        for (Range<LocalDate> range : intervalRangeSet.asDescendingSetOfRanges()) {
            names.add(QueriesExportService.rangeToText(range));
        }

        AbstractWriter writer = exportFormat.getWriter();
        writer.write(names);

        for (PathId id : pathIds) {
            String pathText = idToPathText.get(id);
            if (!StringUtils.isEmpty(pathText)) {
                List<Object> line = new ArrayList<>();
                AccumulatorMap ac = accumulators.computeIfAbsent(id, accumulatorFactory);
                line.add(pathText);
                for (QueryIndicator indicator : indicators) {
                    List<Pair<Range<LocalDate>, Double>> indicatorStat = ac.getIndicator(indicator);
                    Collections.reverse(indicatorStat);
                    for (Pair<Range<LocalDate>, Double> pair : indicatorStat) {
                        line.add(indicator.format(pair.getValue()));
                    }
                }
                writer.write(line);
            }
        }

        try {
            writer.close();
        } catch (IOException e) {
            log.error("Error with close writer", e);
        }
        return writer.getBytes();
    }

    public byte[] getAllIndicatorsForUrl(
            WebmasterHostId hostId, LocalDate userRangeStart, LocalDate userRangeEnd, Integer[] regionId,
            RegionInclusion regionInclusion, SpecialGroup specialGroup, PathId pathId, DeviceType deviceType,
            AggregatePeriod periodParam, FileFormat exportFormat) {
        List<QueryIndicator> indicators = DEFAULT_INDICATORS;
        AggregatePeriod period = periodParam;
        if (period == null) {
            period = AggregatePeriod.DAY;
        }

        final Set<Integer> regionIds = StatisticsHelper.asSet(10, regionId);

        RangeSet<LocalDate> userRangeSet = RangeFactory.singleRange(userRangeStart, userRangeEnd);
        RangeSet<LocalDate> intervalRangeSet = RangeFactory.createRanges(userRangeStart, userRangeEnd, period, false, false);

        List<PathId> pathIds = Collections.singletonList(pathId);

        Range<LocalDate> urlRange = userRangeSet.span().span(userRangeSet.span());
        List<PathStat> dayStat = urlStatisticsService2
                .getStatistics(specialGroup, hostId,
                        indicators, pathIds,
                        regionIds, regionInclusion,
                        urlRange.lowerEndpoint(), urlRange.upperEndpoint(), deviceType);

        List<Object> names = new ArrayList<>();
        names.add("Indicator");
        names.add(QueriesExportService.rangeToText(userRangeSet.span()));
        for (Range<LocalDate> range : intervalRangeSet.asDescendingSetOfRanges()) {
            names.add(QueriesExportService.rangeToText(range));
        }

        AbstractWriter writer = exportFormat.getWriter();
        writer.write(names);

        AccumulatorMap userRangeMap = AccumulatorMap.create(indicators, userRangeSet);
        AccumulatorMap periodMap = AccumulatorMap.create(indicators, intervalRangeSet);
        dayStat.forEach(ds -> {
            userRangeMap.apply(ds);
            periodMap.apply(ds);
        });

        for (QueryIndicator indicator : indicators) {
            List<Object> line = new ArrayList<>();
            line.add(QueriesExportService.translateIndicator(indicator));
            {
                List<Pair<Range<LocalDate>, Double>> userIndicatorStat = userRangeMap.getIndicator(indicator);
                line.add(userIndicatorStat.get(0).getValue());
            }

            {
                List<Pair<Range<LocalDate>, Double>> periodIndicatorStat = periodMap.getIndicator(indicator);
                Collections.reverse(periodIndicatorStat);
                for (Pair<Range<LocalDate>, Double> pair : periodIndicatorStat) {
                    line.add(indicator.format(pair.getValue()));
                }
            }
            writer.write(line);
        }

        try {
            writer.close();
        } catch (IOException e) {
            log.error("Error with close writer", e);
        }
        return writer.getBytes();
    }
}

