package ru.yandex.webmaster3.storage.events.dao;

import java.io.IOException;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.WMCEventType;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;

/**
 * @author avhaliullin
 */
@Slf4j
public class WMCEventMapper {
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new WebmasterJsonModule(false))
            .registerModule(new JodaModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);

    public static String serializeProcessed(WMCEvent event, String observerName, DateTime dateProcessed) {
        WMCEventProcessed ev = new WMCEventProcessed(event, observerName, dateProcessed);
        try {
            return OM.writeValueAsString(ev);
        } catch (JsonProcessingException e) {
            throw new WebmasterException("Failed to serialize user message of type " + event.getContent().getType(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(WMCEventMapper.class,
                            "Failed to serialize user message of type " + event.getContent().getType()), e);
        }
    }

    public static String serializeContent(WMCEvent event) {
        try {
            return OM.writeValueAsString(event.getContent());
        } catch (JsonProcessingException e) {
            throw new WebmasterException("Failed to serialize user message of type " + event.getContent().getType(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(WMCEventMapper.class,
                            "Failed to serialize user message of type " + event.getContent().getType()), e);
        }
    }

    private static JavaType constructType(JsonNode node, WMCEventType eventType) {
        if (eventType == null) {
            log.error("Empty event type for node: {}", node);
        }
        switch (eventType) {
            case RETRANSLATE_TO_USERS:
                JsonNode payloadNode = node.get("payload");
                WMCEventType payloadType = WMCEventType.R.valueOf(node.get("payloadType").asText());
                return OM.getTypeFactory().constructParametrizedType(
                        RetranslateToUsersEvent.class,
                        RetranslateToUsersEvent.class,
                        constructType(payloadNode, payloadType)
                );
            case USER_HOST_MESSAGE:
                JsonNode contentNode = node.get("messageContent");
                MessageTypeEnum contentType = MessageTypeEnum.R.valueOf(contentNode.get("type").asText());
                return OM.getTypeFactory().constructParametrizedType(
                        UserHostMessageEvent.class,
                        UserHostMessageEvent.class,
                        contentType.getMessageClass()
                );
            default:
                return OM.constructType(eventType.getContentClass());
        }
    }

    public static WMCEventContent deserialize(WMCEventType type, String contentString) {
        try {
            JsonNode node = OM.readTree(contentString);
            JavaType eventJavaType = constructType(node, type);
            return OM.readValue(new TreeTraversingParser(node), eventJavaType);
        } catch (IOException e) {
            throw new WebmasterException("Failed to deserialize event of type " + type + " from string: " + contentString,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(WMCEventMapper.class,
                            "Failed to deserialize event of type " + type + " from string: " + contentString), e);
        }
    }

    /**
     * For sending to Logbroker
     */
    private static class WMCEventProcessed {
        public final WMCEventType type;
        public final UUID id;
        public final WMCEventContent content;
        public final String observer;
        public final DateTime created;
        public final DateTime processed;

        WMCEventProcessed(WMCEvent event, String observerName, DateTime dateProcessed) {
            this.type = event.getType();
            this.id = event.getId();
            this.content = event.getContent();
            this.observer = observerName;
            this.created = event.getEventTime();
            this.processed = dateProcessed;
        }
    }
}
