package ru.yandex.webmaster3.storage.events.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.core.logbroker.writer.LogbrokerWriter;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.WMCEventType;
import ru.yandex.webmaster3.storage.events.data.WmcEventContainer;

/**
 * ishalaru
 * 07.12.2020
 **/
@Slf4j
public class LbWMCEventService {
    private static final int MAX_BATCH_SIZE = 25;
    private Map<WMCEventType, LogbrokerWriter> logbrokerClientMap;
    private final BlockingQueue<Pair<WMCEventType, String>> sendQueue = new ArrayBlockingQueue<>(50);
    private ExecutorService sendQueueExecutor;

    public LbWMCEventService(Map<WMCEventType, LogbrokerWriter> logbrokerClientMap) {
        this.logbrokerClientMap = logbrokerClientMap;
        sendQueueExecutor = Executors.newSingleThreadExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("wmcevent-client-queue-%d")
                        .setDaemon(true)
                        .build()
        );
        sendQueueExecutor.execute(new LbWMCEventService.SendWorker());
    }

    public boolean contains(WMCEventType type) {
        return logbrokerClientMap.containsKey(type);
    }

    private void send(WMCEventType type, List<WMCEventContent> content) {
        try {
            final List<String> collect = content.stream().map(JsonMapping::writeValueAsString).collect(Collectors.toList());

            logbrokerClientMap.get(type)
                    .write(JsonMapping.writeValueAsBytes(new WmcEventContainer(type, collect)));
        } catch (Exception e) {
            log.error("Problem in lb sender. Type: " + type, e);
            //throw new WebmasterException("Can't send message in LB.", null, e);
        }
    }

    private void sendInLb(List<Pair<WMCEventType, String>> content) {
        final Map<WMCEventType, List<String>> collect =
                content.stream().collect(W3Collectors.groupByPair());
        for (Map.Entry<WMCEventType, List<String>> item : collect.entrySet()) {
            try {
                logbrokerClientMap.get(item.getKey())
                        .write(JsonMapping.writeValueAsBytes(new WmcEventContainer(item.getKey(), item.getValue())));
            } catch (Exception e) {
                log.error("Problem in lb sender. Type: " + item.getKey(), e);
            }
        }
    }


    public void send(List<WMCEventContent> list) {
        try {
            for (WMCEventContent wmcEventContent : list) {
                if (wmcEventContent == null) {
                    continue;
                }
                boolean enqueued = false;
                try {
                    enqueued = sendQueue.offer(Pair.of(wmcEventContent.getType(), JsonMapping.writeValueAsString(wmcEventContent)), 1, TimeUnit.SECONDS);
                } catch (InterruptedException e) {
                    log.error(e.getMessage(), e);
                }
                if (!enqueued) {
                    log.error("Messages not processed.");
                }
            }
        } catch (Exception exp) {
            log.error(exp.getMessage(), exp);
        }
    }

    public void sendWmcEvent(List<WMCEvent> list) {
        try {
            send(list.stream().map(WMCEvent::getContent).collect(Collectors.toList()));
        } catch (Exception exp) {
            log.error(exp.getMessage(), exp);
        }
    }

    public class SendWorker implements Runnable {
        @Override
        public void run() {
            log.info("Start task");
            while (!Thread.interrupted()) {
                Pair<WMCEventType, String> taskData;
                try {
                    taskData = sendQueue.take();
                } catch (InterruptedException e) {
                    break;
                }
                List<Pair<WMCEventType, String>> taskDataList = new ArrayList<>(MAX_BATCH_SIZE);
                taskDataList.add(taskData);
                sendQueue.drainTo(taskDataList, MAX_BATCH_SIZE - 1);
                sendInLb(taskDataList);

            }
            log.info("Stop task");
        }
    }
}
