package ru.yandex.webmaster3.storage.events2;

import com.google.common.primitives.UnsignedLong;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.events2.HostEvent;
import ru.yandex.webmaster3.core.events2.HostEventData;
import ru.yandex.webmaster3.core.events2.HostEventJsonUtils;
import ru.yandex.webmaster3.core.events2.HostEventType;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.CityHash102;
import ru.yandex.webmaster3.core.events2.events.recheck.SanctionsRecheckRequestedEvent;
import ru.yandex.webmaster3.core.sanctions.SanctionsRecheckRequested;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.SimpleByteArrayOutputStream;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Format;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.From;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.GroupableLimitableOrderable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.OrderBy;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.cases.Case;

import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
public class HostEventsCHDao extends AbstractClickhouseDao implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(HostEventsCHDao.class);

    private static final String DIST_TABLE = "host_events";
    private static final String SHARD_TABLE = "host_events_shard_1";

    public void insertEvents(List<HostEvent> events) throws ClickhouseException {
        Statement st = QueryBuilder.insertInto(DB_WEBMASTER3, SHARD_TABLE)
                .fields(F.DATE, F.TIMESTAMP, F.REQ_ID, F.EVENT_TYPE, F.HOST_ID, F.TASK_ID, F.USER_ID, F.ADMIN_USER_ID, F.DATA)
                .format(Format.Type.TAB_SEPARATED);
        Map<Integer, List<HostEvent>> shard2Events = events.stream().collect(Collectors.groupingBy(this::hash));

        Map<Integer, ClickhouseHost> shard2Host = new HashMap<>();
        for (int shard : shard2Events.keySet()) {
            ClickhouseHost host = getClickhouseServer().pickAliveHostOrFail(shard);
            shard2Host.put(shard, host);
        }
        for (int shard : shard2Events.keySet()) {
            SimpleByteArrayOutputStream bs = new SimpleByteArrayOutputStream();

            for (HostEvent event : shard2Events.get(shard)) {
                bs = packRowValues(bs,
                        toClickhouseDate(event.getDate()),
                        event.getDate().getMillis(),
                        toStringOrEmpty(event.getReqId()),
                        event.getType().value(),
                        event.getHostId().toString(),
                        toStringOrEmpty(event.getTaskId()),
                        zeroOnNull(event.getUserId()),
                        zeroOnNull(event.getAdminUserId()),
                        HostEventJsonUtils.serialize(event.getData())
                );
            }
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults()
                    .setHost(shard2Host.get(shard));
            getClickhouseServer().insert(ctx, st.toString(), bs.toInputStream());
        }
    }

    public List<HostEvent> listEvents(HostEventsFilter filter, PageUtils.Pager pager) throws ClickhouseException {
        List<HostEvent> result = new ArrayList<>();
        forEach(filter, pager, result::add);
        return result;
    }

    private long getHitsAfterDate(LocalDate localDate, HostEventsFilter filter) throws ClickhouseException {
        From from = QueryBuilder.select("count() AS count")
                .from(DB_WEBMASTER3, DIST_TABLE);
        Where where = buildWhere(filter, from);
        where = where.and(QueryBuilder.gt(F.DATE, localDate));
        return queryOne(where, row -> row.getLongUnsafe("count")).orElse(0L);
    }

    private NavigableSet<LocalDate> getDatesForFilter(HostEventsFilter filter, PageUtils.Pager pager) throws ClickhouseException {
        From from = QueryBuilder.select(F.DATE).from(DB_WEBMASTER3, DIST_TABLE);
        Where where = buildWhere(filter, from);

        GroupableLimitableOrderable st = where.orderBy(F.DATE, OrderBy.Direction.DESC);
        if (pager != null) {
            st = st.limit(pager.toRangeStart(), pager.getPageSize());
        }

        return collectAll(st, Collectors.mapping(row -> getLocalDate(row, F.DATE), Collectors.toCollection(TreeSet::new)));
    }

    private int hash(HostEvent event) {
        byte[] hostIdBytes = event.getHostId().toString().getBytes();
        return UnsignedLong.fromLongBits(CityHash102.cityHash64(hostIdBytes, 0, hostIdBytes.length))
                .mod(UnsignedLong.valueOf(getClickhouseServer().getShardsCount())).intValue();
    }

    public void forEach(HostEventsFilter filter, PageUtils.Pager pager, Consumer<HostEvent> consumer) throws ClickhouseException {
        From from = QueryBuilder.select(F.TIMESTAMP, F.REQ_ID, F.EVENT_TYPE, F.HOST_ID, F.TASK_ID, F.USER_ID, F.ADMIN_USER_ID, F.DATA)
                .from(DB_WEBMASTER3, DIST_TABLE);
        Where where = buildWhere(filter, from);
        int offsetCorrection = 0;
        if (filter.getHost() == null) {
            // При указании хоста все работает быстро.
            // Если хоста нет, применяем оптимизацию:
            // 1. Запрос только date вместо * работает быстрее в 10-50 раз
            // 2. Если в исходный запрос явно добавить фильтр по date - он ускорится на порядок
            // 3. При этом пострадает pager - поэтому посмотрим, сколько результатов отбросилось из-за
            //    фильтра по дате, и скорректируем offset
            NavigableSet<LocalDate> dates = getDatesForFilter(filter, pager);
            if (dates.isEmpty()) {
                return;
            }
            where = where.and(QueryBuilder.in(F.DATE, dates));
            // Должно влезать в int, пока в исходный pager не передадут long, а это пока невозможно
            offsetCorrection = Math.toIntExact(getHitsAfterDate(dates.last(), filter));
        }

        GroupableLimitableOrderable st = where.orderBy(F.TIMESTAMP, OrderBy.Direction.DESC);
        if (pager != null) {
            st = st.limit(pager.toRangeStart() - offsetCorrection, pager.getPageSize());
        }

        forEach(st, row -> {
            HostEventType type = HostEventType.R.fromValueOrNull(row.getIntUnsafe(F.EVENT_TYPE));
            if (type == null) {
                consumer.accept(null);
                return;
            }
            HostEventData data = HostEventJsonUtils.deserialize(type, row.getString(F.DATA));
            String taskIdString = getNullableString(row, F.TASK_ID);
            UUID taskId = taskIdString == null ? null : UUID.fromString(taskIdString);
            long ts = row.getLongUnsafe(F.TIMESTAMP);
            Instant timestamp = new Instant(ts);
            HostEvent event = new HostEvent(
                    getNullableString(row, F.REQ_ID),
                    timestamp,
                    taskId,
                    getNullableLong(row, F.USER_ID),
                    getNullableLong(row, F.ADMIN_USER_ID),
                    IdUtils.stringToHostId(row.getString(F.HOST_ID)),
                    data
            );
            consumer.accept(event);
        });
    }

    public long countEvents(HostEventsFilter filter) throws ClickhouseException {
        From from = QueryBuilder.select("count() AS count").from(DB_WEBMASTER3, DIST_TABLE);
        Statement st = buildWhere(filter, from);
        return queryOne(st, row -> row.getLongUnsafe("count")).orElse(0L);
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        long userId = user.getUserId();
        String query = String.format("ALTER TABLE %s ON CLUSTER %s DELETE WHERE user_id=%s",
                DB_WEBMASTER3 + "." + SHARD_TABLE, getClickhouseServer().getClusterId(), userId);
        getClickhouseServer().execute(ClickhouseQueryContext.useDefaults(), query);
    }

    private Where buildWhere(HostEventsFilter filter, From from) {
        Where where = from.where(new Case() {
            @Override
            public String toString() {
                return "1";
            }
        });
        if (filter.getHost() != null) {
            where = where.and(QueryBuilder.eq(F.HOST_ID, filter.getHost().toString()));
        }
        if (filter.getRequestId() != null) {
            where = where.and(QueryBuilder.eq(F.REQ_ID, filter.getRequestId()));
        }
        if (filter.getTaskId() != null) {
            where = where.and(QueryBuilder.eq(F.TASK_ID, filter.getTaskId().toString()));
        }
        if (filter.getUserId() != null) {
            where = where.and(QueryBuilder.eq(F.USER_ID, filter.getUserId()));
        }
        if (filter.getAdminUserId() != null) {
            where = where.and(QueryBuilder.eq(F.ADMIN_USER_ID, filter.getAdminUserId()));
        }
        if (filter.getDateFrom() != null) {
            where = where.and(QueryBuilder.gte(F.TIMESTAMP, filter.getDateFrom().getMillis()));
        }
        if (filter.getDateTo() != null) {
            where = where.and(QueryBuilder.lte(F.TIMESTAMP, filter.getDateTo().getMillis()));
        }
        if (filter.getEventTypes() != null && !filter.getEventTypes().isEmpty()) {
            where = where.and(
                    QueryBuilder.in(
                            F.EVENT_TYPE,
                            filter.getEventTypes().stream().map(HostEventType::value).collect(Collectors.toList())
                    )
            );
        }
        return where;
    }

    private static class F {
        static final String DATE = "date";
        static final String TIMESTAMP = "timestamp";
        static final String REQ_ID = "req_id";
        static final String EVENT_TYPE = "event_type";
        static final String HOST_ID = "host_id";
        static final String TASK_ID = "task_id";
        static final String USER_ID = "user_id";
        static final String ADMIN_USER_ID = "admin_user_id";
        static final String DATA = "data";
    }
}
