package ru.yandex.webmaster3.storage.events2;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.events2.HostEventType;

import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Set;
import java.util.UUID;

/**
 * @author avhaliullin
 */
public class HostEventsFilter {
    private final WebmasterHostId host;
    private final String requestId;
    private final UUID taskId;
    private final Long userId;
    private final Long adminUserId;
    private final Instant dateFrom;
    private final Instant dateTo;
    private final Set<HostEventType> eventTypes;

    public HostEventsFilter(WebmasterHostId host, String requestId, UUID taskId, Long userId, Long adminUserId,
                            Instant dateFrom, Instant dateTo, Set<HostEventType> eventTypes) {
        this.host = host;
        this.requestId = requestId;
        this.taskId = taskId;
        this.userId = userId;
        this.adminUserId = adminUserId;
        this.dateFrom = dateFrom;
        this.dateTo = dateTo;
        this.eventTypes = eventTypes;
    }

    public WebmasterHostId getHost() {
        return host;
    }

    public String getRequestId() {
        return requestId;
    }

    public UUID getTaskId() {
        return taskId;
    }

    public Long getUserId() {
        return userId;
    }

    public Long getAdminUserId() {
        return adminUserId;
    }

    public Instant getDateFrom() {
        return dateFrom;
    }

    public Instant getDateTo() {
        return dateTo;
    }

    public Set<HostEventType> getEventTypes() {
        return eventTypes;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private WebmasterHostId host;
        private String requestId;
        private UUID taskId;
        private Long userId;
        private Long adminUserId;
        private Instant dateFrom;
        private Instant dateTo;
        private Set<HostEventType> eventTypes;

        public Builder host(WebmasterHostId hostId) {
            this.host = hostId;
            return this;
        }

        public Builder requestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        public Builder taskId(UUID taskId) {
            this.taskId = taskId;
            return this;
        }

        public Builder userId(Long userId) {
            this.userId = userId;
            return this;
        }

        public Builder adminUserId(Long adminUserId) {
            this.adminUserId = adminUserId;
            return this;
        }

        public Builder dateFrom(Instant dateFrom) {
            this.dateFrom = dateFrom;
            return this;
        }

        public Builder dateTo(Instant dateTo) {
            this.dateTo = dateTo;
            return this;
        }

        public Builder dateFrom(DateTime dateFrom) {
            if (dateFrom == null) {
                this.dateFrom = null;
            } else {
                this.dateFrom = dateFrom.toInstant();
            }
            return this;
        }

        public Builder dateTo(DateTime dateTo) {
            if (dateTo == null) {
                this.dateTo = null;
            } else {
                this.dateTo = dateTo.toInstant();
            }
            return this;
        }

        public Builder eventTypes(Collection<HostEventType> eventTypes) {
            if (eventTypes == null) {
                this.eventTypes = null;
            } else {
                this.eventTypes = EnumSet.copyOf(eventTypes);
            }
            return this;
        }

        public Builder eventTypes(HostEventType eventType1, HostEventType... eventTypes) {
            this.eventTypes = EnumSet.of(eventType1, eventTypes);
            return this;
        }

        public HostEventsFilter build() {
            return new HostEventsFilter(host, requestId, taskId, userId, adminUserId, dateFrom, dateTo, eventTypes);
        }
    }
}
