package ru.yandex.webmaster3.storage.favicon;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemStorageType;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostGeneration;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsProvider;

/**
 * Created by Oleg Bazdyrev on 03/08/2020.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class FaviconProblemsProvider implements SiteProblemsProvider {

    private static Set<SiteProblemTypeEnum> SUPPORTED_PROBLEM_TYPES =
            Collections.unmodifiableSet(Stream.of(SiteProblemTypeEnum.values())
            .filter(type -> type.getStorageType() == SiteProblemStorageType.FAVICON_PROBLEM)
            .filter(type -> !type.isDisabled())
            .collect(Collectors.toCollection(() -> EnumSet.noneOf(SiteProblemTypeEnum.class))));

    private final HostFaviconsCHDao mdbHostFaviconsCHDao;

    @Override
    public @NotNull List<? extends AbstractProblemInfo> listProblems(@NotNull WebmasterHostId hostId,
                                                                     @Nullable UUID generationId) {
        return mdbHostFaviconsCHDao.getProblems(hostId);
    }

    @Override
    public Map<WebmasterHostId, List<? extends AbstractProblemInfo>> listProblems(List<WebmasterHostGeneration> hosts) {
        return mdbHostFaviconsCHDao.getProblems(hosts.stream().map(WebmasterHostGeneration::getHostId).collect(Collectors.toList()));
    }

    @Override
    public @NotNull Set<SiteProblemTypeEnum> getSupportedProblemTypes() {
        return SUPPORTED_PROBLEM_TYPES;
    }
}
