package ru.yandex.webmaster3.storage.favicon;

import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.protobuf.InvalidProtocolBufferException;
import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.proto.Favicons;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.data.RealTimeSiteProblemInfo;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHRow;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * Created by Oleg Bazdyrev on 03/08/2020.
 */
public class HostFaviconsCHDao extends AbstractClickhouseDao {

    public static final CHTable TABLE = CHTable.builder()
            .database(AbstractClickhouseDao.DB_WEBMASTER3_CHECKLIST)
            .name("host_favicons_" + "%s")
            .partitionBy("toYYYYMM(" + F.DATE + ")")
            .keyField(F.DATE, CHPrimitiveType.Date)
            .keyField(F.HOST_ID, CHPrimitiveType.String)
            .field(F.FAVICONS, CHPrimitiveType.String)
            .sharded(false)
            .build();

    @Setter
    private TableProvider tableStorage;


    @NotNull
    private static Pair<WebmasterHostId, Favicons.THostFaviconsAndProblems> mapper(CHRow row) {
        try {
            WebmasterHostId hostId = row.getHostId(F.HOST_ID);
            byte[] data = Base64.getDecoder().decode(row.getBytes(F.FAVICONS));
            return Pair.of(hostId, Favicons.THostFaviconsAndProblems.parseFrom(data));
        } catch (InvalidProtocolBufferException e) {
            throw new WebmasterException("Error reading protobuf favicons from clickhouse",
                    new WebmasterErrorResponse.ClickhouseErrorResponse(HostFaviconsCHDao.class, null, e), e);
        }
    }

    private Map<WebmasterHostId, Favicons.THostFaviconsAndProblems> getFaviconsAndProblems(Collection<WebmasterHostId> hostIds) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.HOST_FAVICONS);

        Where st = QueryBuilder.select(F.HOST_ID, F.FAVICONS)
                .from(table.getLocalTableName())
                .where(QueryBuilder.in(F.HOST_ID, hostIds));

        return getClickhouseServer().collectAll(st.toString(),
                Collectors.mapping(HostFaviconsCHDao::mapper, W3Collectors.toHashMap())
        );
    }

    @NotNull
    private Optional<Favicons.THostFaviconsAndProblems> getFaviconsAndProblems(WebmasterHostId hostId) {
        ClickhouseTableInfo table = tableStorage.getTable(TableType.HOST_FAVICONS);

        Where st = QueryBuilder.select(F.HOST_ID, F.FAVICONS)
                .from(table.getLocalTableName())
                .where(QueryBuilder.eq(F.HOST_ID, hostId));

        return getClickhouseServer().queryOne(st.toString(), row -> mapper(row).getValue());
    }

    public List<FaviconInfo> getFavicons(WebmasterHostId hostId) {
        return getFaviconsAndProblems(hostId).map(faviconsAndProblems -> faviconsAndProblems.getFaviconsList().stream().map(FaviconInfo::fromProto)
                .collect(Collectors.toList())).orElse(Collections.emptyList());
    }

    public Map<WebmasterHostId, List<FaviconInfo>> getFavicons(Collection<WebmasterHostId> hostIds) {
        Map<WebmasterHostId, Favicons.THostFaviconsAndProblems> faviconsAndProblemsByHostId
                = getFaviconsAndProblems(hostIds);
        Map<WebmasterHostId, List<FaviconInfo>> result = new HashMap<>();
        for (var entry : faviconsAndProblemsByHostId.entrySet()) {
            WebmasterHostId hostId = entry.getKey();
            List<FaviconInfo> faviconInfos = entry.getValue().getFaviconsList().stream().map(FaviconInfo::fromProto)
                    .collect(Collectors.toList());
            result.put(hostId, faviconInfos);
        }
        hostIds.forEach(host -> result.putIfAbsent(host, Collections.emptyList()));
        return result;
    }

    public List<RealTimeSiteProblemInfo> getProblems(WebmasterHostId hostId) {
        return getFaviconsAndProblems(hostId)
                .map(faviconsAndProblems -> getFaviconProblems(hostId, faviconsAndProblems))
                .orElse(Collections.emptyList());
    }

    public Map<WebmasterHostId, List<? extends AbstractProblemInfo>> getProblems(Collection<WebmasterHostId> hostIds) {
        var faviconsAndProblemsByHostId = getFaviconsAndProblems(hostIds);

        Map<WebmasterHostId, List<? extends AbstractProblemInfo>> result = new HashMap<>();
        for (var entry : faviconsAndProblemsByHostId.entrySet()) {
            result.put(entry.getKey(), getFaviconProblems(entry.getKey(), entry.getValue()));
        }
        hostIds.forEach(host -> result.putIfAbsent(host, Collections.emptyList()));

        return result;
    }

    @NotNull
    private List<RealTimeSiteProblemInfo> getFaviconProblems(WebmasterHostId hostId,
                                                             Favicons.THostFaviconsAndProblems faviconsAndProblems) {
        List<RealTimeSiteProblemInfo> result = new ArrayList<>();
        for (var problem : faviconsAndProblems.getProblemsList()) {
            switch (problem.getType()) {
                case MISSING_FAVICON:
                    result.add(new RealTimeSiteProblemInfo(hostId, new DateTime(problem.getLastUpdate()),
                            new DateTime(problem.getActualSince()),
                            new DateTime(problem.getLastUpdate()), SiteProblemState.PRESENT,
                            SiteProblemTypeEnum.MISSING_FAVICON,
                            new SiteProblemContent.MissingFavicon(), 0));
                    break;
                case FAVICON_ERROR:
                    result.add(new RealTimeSiteProblemInfo(hostId, new DateTime(problem.getLastUpdate()),
                            new DateTime(problem.getActualSince()),
                            new DateTime(problem.getLastUpdate()), SiteProblemState.PRESENT,
                            SiteProblemTypeEnum.FAVICON_ERROR,
                            new SiteProblemContent.FaviconError(), 0));
                    break;
                case BIG_FAVICON_ABSENT:
                    result.add(new RealTimeSiteProblemInfo(hostId, new DateTime(problem.getLastUpdate()),
                            new DateTime(problem.getActualSince()),
                            new DateTime(problem.getLastUpdate()), SiteProblemState.PRESENT,
                            SiteProblemTypeEnum.BIG_FAVICON_ABSENT,
                            new SiteProblemContent.BigFaviconAbsent(), 0));

                    break;
            }
        }
        return result;
    }

    public List<FaviconInfo> getFaviconsSamples(WebmasterHostId hostId, SiteProblemTypeEnum problemType) {
        return getFaviconsAndProblems(hostId).map(faviconsAndProblems -> faviconsAndProblems.getProblemsList().stream()
                .filter(problem -> problem.getType().name().equals(problemType.name()))
                .map(problem -> problem.getSamplesList().stream().map(FaviconInfo::fromProto).collect(Collectors.toList()))
                .findAny().orElse(Collections.emptyList())
        ).orElse(Collections.emptyList());
    }


    private interface F {
        String DATE = "date";
        String HOST_ID = "host_id";
        String FAVICONS = "favicons";
    }
}
