package ru.yandex.webmaster3.storage.feeds;

import java.util.Collection;
import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorExample;
import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorInfo;
import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorStatus;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;


/**
 * @author kravchenko99
 * @date 1/19/22
 */

@Repository
public class FeedsDefectRateErrorYDao extends AbstractYDao {
    private static final String TABLE_NAME = "defect_rate_error";

    public FeedsDefectRateErrorYDao() {
        super(PREFIX_FEEDS, TABLE_NAME);
    }

    public void batchInsert(Collection<FeedsDefectRateErrorInfo> items) {
        DateTime now = DateTime.now();
        List<Pair<FeedsDefectRateErrorInfo, DateTime>> collect = items.stream().map(x -> Pair.of(x, now)).toList();
        batchInsert(VALUE_MAPPER, collect).execute();

        delete().where(F.TIME_INSERT.lt(now)).execute();
    }

    public List<FeedsDefectRateErrorInfo> list(String domain) {
        return select(MAPPER).where(F.DOMAIN.eq(domain)).queryForList();
    }

    private static final DataMapper<FeedsDefectRateErrorInfo> MAPPER = DataMapper.create(
            F.DATE, F.DOMAIN, F.TYPE, F.STATUS, F.SEVERITY, F.EXAMPLES, FeedsDefectRateErrorInfo::new
    );

    private static final ValueDataMapper<Pair<FeedsDefectRateErrorInfo, DateTime>> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.DOMAIN, x -> x.getKey().getDomain()),
            Pair.of(F.TYPE, x -> x.getKey().getType()),
            Pair.of(F.STATUS, x -> x.getKey().getStatus()),
            Pair.of(F.SEVERITY, x -> x.getKey().getSeverity()),
            Pair.of(F.EXAMPLES, x -> x.getKey().getExamples()),
            Pair.of(F.TIME_INSERT, Pair::getValue)
    );

    private interface F {
        TypeReference<List<FeedsDefectRateErrorExample>> LIST = new TypeReference<>() {
        };

        //key
        Field<LocalDate> DATE = Fields.jodaDateField("date").makeOptional();
        Field<String> DOMAIN = Fields.stringField("domain");
        Field<NativeFeedType> TYPE = Fields.intEnumField("type", NativeFeedType.R);

        Field<FeedsDefectRateErrorStatus> STATUS = Fields.stringEnumField("status", EnumResolver.er(FeedsDefectRateErrorStatus.class));
        Field<FeedsErrorSeverity> SEVERITY = Fields.stringEnumField("severity", EnumResolver.er(FeedsErrorSeverity.class));
        Field<List<FeedsDefectRateErrorExample>> EXAMPLES = Fields.jsonField2("examples", LIST).makeOptional();
        Field<DateTime> TIME_INSERT = Fields.jodaDateTimeField("time_insert");
    }
}
