package ru.yandex.webmaster3.storage.feeds;

import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorExample;
import ru.yandex.webmaster3.core.feeds.feed.FeedsOfferBanInfo;
import ru.yandex.webmaster3.core.feeds.feed.FeedsOfferBanTypeEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;


/**
 * @author kravchenko99
 * @date 1/19/22
 */

@Repository
public class FeedsOfferBansYDao extends AbstractYDao {
    private static final String TABLE_NAME = "feeds_offer_bans";

    public FeedsOfferBansYDao() {
        super(PREFIX_FEEDS, TABLE_NAME);
    }


    public void deleteBefore(DateTime dt) {
        delete().where(F.TIME_INSERT.lt(dt)).execute();
    }

    public int count(String domain, NativeFeedType type) {
        return select(DataMapper.SINGLE_COLUMN_INT_MAPPER).countAll().where(F.DOMAIN.eq(domain)).and(F.TYPE.eq(type)).queryOne();
    }

    public Map<NativeFeedType, Long> countByType(String domain) {
        return select(COUNT_MAPPER).where(F.DOMAIN.eq(domain)).groupBy(F.TYPE).queryForList().stream().collect(W3Collectors.toEnumMap(NativeFeedType.class));
    }

    public List<FeedsOfferBanInfo> list(String domain, NativeFeedType type) {
        return select(MAPPER).where(F.DOMAIN.eq(domain)).and(F.TYPE.eq(type)).queryForList(
                Pair.of(F.URL, FeedsOfferBanInfo::getUrl),
                Pair.of(F.OFFER_ID, FeedsOfferBanInfo::getOfferId)
        );
    }

    private static final DataMapper<FeedsOfferBanInfo> MAPPER = DataMapper.create(
            F.DOMAIN, F.URL, F.OFFER_ID, F.BAN_REASON, F.BAN_TIMESTAMP, F.COMMENT, FeedsOfferBanInfo::new
    );

    private static final DataMapper<Pair<NativeFeedType, Long>> COUNT_MAPPER = DataMapper.createMapper(
            r -> Pair.of(NativeFeedType.R.fromValue(r.getColumn(1).getInt32()), r.getColumn(0).getUint64()),
            F.TYPE, Fields.unsignedLongField("count(*)")
    );

    private interface F {
        TypeReference<List<FeedsDefectRateErrorExample>> LIST = new TypeReference<>() {
        };

        //key
        Field<String> DOMAIN = Fields.stringField("domain");
        Field<NativeFeedType> TYPE = Fields.intEnumField("type", NativeFeedType.R);
        Field<String> URL = Fields.stringField("url");
        Field<String> OFFER_ID = Fields.stringField("offer_id");

        Field<FeedsOfferBanTypeEnum> BAN_REASON =
                Fields.intEnumField("ban_reason", FeedsOfferBanTypeEnum.R);

        Field<DateTime> BAN_TIMESTAMP = Fields.jodaDateTimeField("ban_timestamp");
        Field<String> COMMENT = Fields.stringField("comment").makeOptional();

        Field<DateTime> TIME_INSERT = Fields.jodaDateTimeField("time_insert");
    }
}
