package ru.yandex.webmaster3.storage.feeds;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Triple;
import org.apache.http.entity.ContentType;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.W3RegionInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.feeds.feed.FeedServiceType;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.feeds.feed_domain.FeedDomainInfo;
import ru.yandex.webmaster3.core.feeds.mbi.DatacampPartnerProperties;
import ru.yandex.webmaster3.core.feeds.mbi.FeedRegisterUrlRequest;
import ru.yandex.webmaster3.core.feeds.mbi.MbiService;
import ru.yandex.webmaster3.core.feeds.mbi.RegisterFeedResponse;
import ru.yandex.webmaster3.core.feeds.mbi.UpdateFeedFeaturesRequest;
import ru.yandex.webmaster3.core.regions.RegionUtils;
import ru.yandex.webmaster3.core.regions.W3RegionsTreeService;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.zora.GoZoraService;
import ru.yandex.webmaster3.core.zora.go_data.request.GoZoraRequest;
import ru.yandex.webmaster3.core.zora.go_data.response.GoZoraResponseFetchUrl;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserDomainMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.storage.turbo.dao.TurboFeedsSettingsYDao;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

import static ru.yandex.webmaster3.core.WebmasterCommonErrorType.INTERNAL__GO_ZORA_TIMED_OUT_ERROR;
import static ru.yandex.webmaster3.storage.feeds.FeedsService.RemoveFeedStatus.NOT_EXIST;
import static ru.yandex.webmaster3.storage.feeds.FeedsService.RemoveFeedStatus.OK;

/**
 * @author kravchenko99
 * @date 8/5/21
 */

@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Service
public class FeedsService implements UserTakeoutDataProvider {
    public static final int LIMIT_FEEDS_ON_DOMAIN = 5000;

    public static final int RUSSIA = 225;
    public static final int MOSCOW = 213;
    private static final Set<String> ACCEPTABLE_MIME_TYPES = Set.of(
            ContentType.APPLICATION_XML.getMimeType(),
            ContentType.APPLICATION_OCTET_STREAM.getMimeType(),
            ContentType.TEXT_XML.getMimeType(),
            "application/gzip",
            "application/x-gzip"
//          "application/zip"
    );
    public record UrlWithUserInfo(String url, String login, String password){}

    private final WMCEventsService wmcEventsService;
    private final FeedsNative2YDao feedsNative2YDao;
    private final MbiService mbiService;
    private final FeedsDomainInfoYDao feedsDomainInfoYDao;
    private final TurboFeedsSettingsYDao turboFeedsSettingsYDao;
    private final GoZoraService goZoraService;
    private final W3RegionsTreeService w3regionsTreeService;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;

    @Nullable
    public static URL canonizeUrl(String requestUrl) {
        try {
            return new URL(requestUrl);
        } catch (MalformedURLException e) {
            return null;
        }
    }

    public boolean isRussiaRegion(int regionId) {
        if (regionId ==RUSSIA) {
            return true;
        }
        Optional<W3RegionInfo> exactRegionInfo = Optional.ofNullable(w3regionsTreeService.getExactRegionInfo(regionId));
        boolean test = exactRegionInfo.map(RegionUtils.VISIBILITY_PREDICATE::test).orElse(false);
        if (!test) {
            return false;
        }
        W3RegionInfo visibleParent = w3regionsTreeService.getVisibleParent(regionId, x -> x.getType() == 3);
        return visibleParent != null && visibleParent.getId()==RUSSIA;
    }

    public NativeFeedStatus getStatus(NativeFeedType type,
                                      FeedsOffersLogsHistoryCHDao.FeedRecord offerState,
                                      FeedsOffersLogsHistoryCHDao.FeedRecord serpdataState) {
        FeedStats offerStats = Optional.ofNullable(offerState).map(FeedsOffersLogsHistoryCHDao.FeedRecord::getStats).orElse(FeedStats.EMPTY);
        FeedStats errorStats = Optional.ofNullable(offerState).map(FeedsOffersLogsHistoryCHDao.FeedRecord::getErrorStats).orElse(FeedStats.EMPTY);
        FeedStats serpdataStats = Optional.ofNullable(serpdataState).map(FeedsOffersLogsHistoryCHDao.FeedRecord::getErrorStats).orElse(FeedStats.EMPTY);

        NativeFeedStatus status;
        if (type == NativeFeedType.STORES) {
            status = Optional.ofNullable(offerState).map(FeedsOffersLogsHistoryCHDao.FeedRecord::getStatus).orElse(NativeFeedStatus.IN_PROGRESS);
        } else {
            status = FeedStats.calcFeedStatus(offerStats, errorStats, serpdataStats);
        }
        return status;
    }

    // получаем для товарных фидов, домены которые в поиске или частично в поиске
    public Set<String> getDomainsWithTvInSearch(List<String> domains) {
        List<NativeFeedInfo2> domainsWithSuccessfulGoodFeeds = feedsNative2YDao.hasSuccessfulGoodsFeeds(domains);
        List<Triple<Long, Long, Long>> tvIds = domainsWithSuccessfulGoodFeeds.stream()
                .map(x -> Triple.of(x.getBusinessId(), x.getPartnerId(), x.getFeedId()))
                .toList();
        Map<Triple<Long, Long, Long>, String> tvIdToDomain = domainsWithSuccessfulGoodFeeds.stream()
                .collect(Collectors.toMap(x -> Triple.of(x.getBusinessId(), x.getPartnerId(), x.getFeedId()),
                        NativeFeedInfo2::getDomain));

        return goodsOffersLogsHistoryCHDao.getLastState(tvIds)
                .stream()
                .filter(x -> x.getStatus().getFeedStatus() == NativeFeedStatus.WARNING ||
                        x.getStatus().getFeedStatus() == NativeFeedStatus.SUCCESS)
                .map(x -> tvIdToDomain.get(Triple.of(x.getBusinessId(), x.getPartnerId(), x.getFeedId())))
                .collect(Collectors.toSet());
    }

    public RemoveFeedStatus removeFeed(String domain, String url) {

        // temp parse url + login + password
        FeedsService.UrlWithUserInfo urlWithUserInfo = parseUrlWithoutError(url);
        if (urlWithUserInfo == null) {
            return NOT_EXIST;
        }
        url = urlWithUserInfo.url();
        //

        var info = feedsNative2YDao.get(domain, url);
        if (info == null) {
            return RemoveFeedStatus.NOT_EXIST;
        }
        if (info.getPartnerId() != null) {
            if (info.getStatusScc() != NativeFeedSccStatus.BANNED) {
                mbiService.removeShop(info.getPartnerId());
            }
        }

        feedsNative2YDao.remove(domain, url);
        return OK;
    }

    public boolean containsFeed(String url, NativeFeedType type) {
        return feedsNative2YDao.containsFeed(url) || (type != NativeFeedType.STORES && turboFeedsSettingsYDao.containsFeed(url));
    }

    public void addFeed(
            String domain,
            String url,
            List<Integer> regionIds,
            long userId,
            String login,
            String password,
            NativeFeedType type,
            Set<FeedServiceType> enabledServices) {
        if (type == NativeFeedType.STORES) {
            addMbiFeed(domain, url, regionIds,
                    userId, login, password,
                    type, enabledServices);
        } else {
            addUnisearchFeed(domain, url, regionIds,
                    userId, login, password,
                    type, enabledServices);
        }
    }
    public NativeFeedInfo2 findById(long businessId, long partnerId) {
        FeedDomainInfo byId = feedsDomainInfoYDao.findById(businessId);
        if (byId == null) {
            return null;
        }
        String domain = byId.getDomain();
        return feedsNative2YDao.getPartnerId(domain, partnerId);
    }

    public TurboFeedSettings findTurboById(long businessId, long partnerId) {
        FeedDomainInfo byId = feedsDomainInfoYDao.findById(businessId);
        if (byId == null) {
            return null;
        }
        String domain = byId.getDomain();
        return turboFeedsSettingsYDao.getFeeds(domain).stream()
                .filter(x -> x.getDomain().equals(domain) && Objects.equals(x.getPartnerId(), partnerId)).findFirst().orElse(null);
    }

    //url + login + password
    public UrlWithUserInfo parseUrl(String  url) {
        try {
            return parseUrl(new URL(url));
        } catch (MalformedURLException e) {
            throw new IllegalArgumentException();
        }
    }

    //url + login + password
    public UrlWithUserInfo parseUrlWithoutError(String  url) {
        try {
            return parseUrl(url);
        } catch (IllegalArgumentException e){
            return null;
        }
    }

    public UrlWithUserInfo parseUrl(URL url) {
        String userInfo = url.getUserInfo();
        String urlStr;
        String login;
        String password;

        if (userInfo != null) {
            String[] split = userInfo.split(":", 2);
            login = split[0];
            if (split.length == 2) {
                password = split[1];
            } else {
                password = null;
            }

            String file = url.getRef() == null ? url.getFile() : url.getFile() + "#" + url.getRef();
            try {
                urlStr = new URL(url.getProtocol(), url.getHost(), url.getPort(), file).toString();
            } catch (MalformedURLException e) {
                throw new IllegalArgumentException(e);
            }
        } else {
            urlStr = url.toString();
            login = null;
            password = null;
        }
        return new UrlWithUserInfo(urlStr, login, password);
    }

    public void addUnisearchFeed(String domain,
                                 String url,
                                 List<Integer> regionsId,
                                 long userId,
                                 String login,
                                 String password,
                                 NativeFeedType type,
                                 Set<FeedServiceType> enabledServiceTypes) {
        DateTime now = DateTime.now();
        var nativeFeedInfo = NativeFeedInfo2.ofUs(
                domain, url, userId, regionsId,
                login, password,
                NativeFeedStatus.IN_PROGRESS, List.of(),
                NativeFeedSccStatus.IN_PROGRESS, List.of(),
                type,
                enabledServiceTypes,
                now, now, true, now, NativeFeedStatus.SUCCESS);
        feedsNative2YDao.update(nativeFeedInfo);
    }

    public void addMbiFeed(String domain,
                           String url,
                           List<Integer> regionsId,
                           long userId,
                           String login,
                           String password,
                           NativeFeedType type,
                           Set<FeedServiceType> enabledServiceTypes) {
        RegisterFeedResponse registerFeedResponse = registerMbiFeed(domain, url, regionsId, userId, login, password);

        DateTime now = DateTime.now();
        NativeFeedInfo2 nativeFeedInfo = NativeFeedInfo2.ofMbi(
                domain, url, userId, regionsId,
                login, password,
                NativeFeedStatus.IN_PROGRESS, List.of(),
                NativeFeedSccStatus.IN_PROGRESS, List.of(),
                type,
                enabledServiceTypes,
                now, now, true,
                registerFeedResponse.getBusinessId(),
                registerFeedResponse.getPartnerId(),
                registerFeedResponse.getFeedId(),
                NativeFeedStatus.IN_PROGRESS
        );
        feedsNative2YDao.update(nativeFeedInfo);
    }

    @NotNull
    public ZoraCheckResult checkUrl(String url, String login, String password) {
        GoZoraResponseFetchUrl zoraResponse;
        try {
            zoraResponse = goZoraService.executeRequestFetchResponse(new GoZoraRequest(url, false, true, false, login, password));
        } catch (WebmasterException e) {
            log.error("GoZora - {}", e.getMessage(), e);
            if (e.getError().getCode() == INTERNAL__GO_ZORA_TIMED_OUT_ERROR) {
                return ZoraCheckResult.TIMED_OUT;
            }
            throw e;
        }

        if (zoraResponse.getHttpCode() / 100 != 2) {
            return ZoraCheckResult.BAD_CODE;
        }

        if (zoraResponse.getMimeType() == null || !ACCEPTABLE_MIME_TYPES.contains(zoraResponse.getMimeType())) {
            return ZoraCheckResult.BAD_MIME_TYPE;
        }
        return ZoraCheckResult.OK;
    }

    public enum ZoraCheckResult {
        OK,
        BAD_CODE,
        BAD_MIME_TYPE,
        TIMED_OUT,
        BANNED
        ;
    }


    @NotNull
    public RegisterFeedResponse registerMbiFeed(String domain,
                                                 String url,
                                                 List<Integer> regionsId,
                                                 long userId,
                                                 String login,
                                                 String password) {
        var feedDomainInfo = feedsDomainInfoYDao.get(domain);
        Long businessId = feedDomainInfo.map(FeedDomainInfo::getBusinessId).orElse(null);

        DatacampPartnerProperties properties = new DatacampPartnerProperties(true, domain, regionsId);

        FeedRegisterUrlRequest feedRegisterUrlRequest = new FeedRegisterUrlRequest(url, userId,
                businessId, login, password, properties);

        RegisterFeedResponse registerFeedResponse = mbiService.addUrl(feedRegisterUrlRequest);
        UpdateFeedFeaturesRequest updateFeedFeaturesRequest =
                new UpdateFeedFeaturesRequest(registerFeedResponse.getPartnerId(), properties);
        if (feedDomainInfo.isEmpty()) {
            feedsDomainInfoYDao.update(FeedDomainInfo.of(domain, registerFeedResponse.getBusinessId()));
        }
        mbiService.updateFeatures(updateFeedFeaturesRequest);
        return registerFeedResponse;
    }

    public void sendFailedSccMessage(String domain, String feedUrl) {
        sendFeedsStatusMessage(domain, feedUrl, MessageTypeEnum.FEEDS_SCC_FAILED);
    }

    public void sendBannedSccMessage(String domain, String feedUrl) {
        sendFeedsStatusMessage(domain, feedUrl, MessageTypeEnum.FEEDS_SCC_BANNED);
    }

    public void sendPingerDisableMessage(String domain, String feedUrl) {
        sendFeedsStatusMessage(domain, feedUrl, MessageTypeEnum.FEEDS_PINGER_DISABLE);
    }

    public void sendPingerEnableMessage(String domain, String feedUrl) {
        sendFeedsStatusMessage(domain, feedUrl, MessageTypeEnum.FEEDS_PINGER_ENABLE);
    }

    public void sendFeedsStatusMessage(String domain, String feedUrl, MessageTypeEnum messageType) {
        WebmasterHostId hostId = IdUtils.urlToHostId(domain);

        var messageContent = switch (messageType) {
            case FEEDS_SCC_BANNED -> new MessageContent.FeedsSccBanned(hostId, feedUrl);
            case FEEDS_SCC_FAILED -> new MessageContent.FeedsSccFailed(hostId, feedUrl);
            case FEEDS_PINGER_DISABLE -> new MessageContent.FeedsPingerDisable(hostId, feedUrl);
            case FEEDS_PINGER_ENABLE -> new MessageContent.FeedsPingerEnable(hostId, feedUrl);
            default -> throw new IllegalArgumentException("Expected message type FEEDS_*");
        };

        wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserDomainMessageEvent.create(
                hostId,
                messageContent,
                NotificationType.FEEDS_INFO,
                false)
        )));
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        feedsNative2YDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(feedsNative2YDao.getTablePath());
    }

    public static enum RemoveFeedStatus {
        OK,
        NOT_EXIST
    }

}
