package ru.yandex.webmaster3.storage.geoadv;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.google.common.base.Charsets;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.cache.dao.GeoAdvCacheYDao;
import ru.yandex.webmaster3.storage.services.CompanyInfo;
import ru.yandex.webmaster3.storage.services.ServiceInfo;
import ru.yandex.webmaster3.storage.services.SiteServiceType;
import ru.yandex.webmaster3.storage.services.SiteServicesCHDao;

@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class GeoAdvApiService extends AbstractExternalAPIService {

    private static final int SOCKET_TIMEOUT = 10_000;
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String CALC_PRICE_METHOD = "/priority/v2/price";
    private static final String GET_COMPANY_METHOD = "/priority/v1/get-company?id=";

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(1, Duration.standardSeconds(5));

    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new WebmasterJsonModule(false))
            .registerModule(new JodaModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);

    private CloseableHttpClient httpClient;

    private final SiteServicesCHDao mdbSiteServicesCHDao;
    private final GeoAdvCacheYDao geoAdvCacheYDao;

    private final String getGeoAdvUrl;
    private final String oauthToken;

    public void init() {

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(10, TimeUnit.MINUTES)
                .build();
    }


    public GeoAdvertisingInfo getGeoAdv(@NotNull WebmasterHostId hostId) {
        GeoAdvertisingInfo value = geoAdvCacheYDao.get(hostId);
        if (value != null) {
            return value;
        }

        value = getGeoAdvByHost(hostId);
        geoAdvCacheYDao.add(hostId, value);
        return value;
    }

    private GeoAdvertisingInfo getGeoAdvByHost(@NotNull WebmasterHostId hostId) {
        String domain = WwwUtil.cutWWWAndM(hostId.getPunycodeHostname());
        ServiceInfo serviceInfo = mdbSiteServicesCHDao.getSiteServiceInfo(domain).get(SiteServiceType.COMPANY_INFO);

        if (serviceInfo != null) {
            CompanyInfo companyInfo = (CompanyInfo) serviceInfo;
            List<Long> permalinks = companyInfo.getPermalink();

            if (permalinks != null && permalinks.size() == 1) {
                Long permalink = permalinks.get(0);

                //check active company
                GeoCompanyInfo company = getCompanyInfo(permalink);
                if (company.getGeoAdvertisingResponse() != null && !Boolean.FALSE.equals(company.getGeoAdvertisingResponse().getHasActiveCampaigns())) {
                    return new GeoAdvertisingInfo(); //empty response
                }

                return getCompanyAdvertising(permalink);
            }
        }

        return new GeoAdvertisingInfo(); //empty response
    }

    private GeoAdvertisingInfo getCompanyAdvertising(Long companyId) {
        GeoAdvertisingInfo advertisingInfo = new GeoAdvertisingInfo();
        advertisingInfo.setPermalink(companyId);

        HttpPost post = new HttpPost(getGeoAdvUrl + CALC_PRICE_METHOD);

        post.setEntity(new StringEntity(JsonMapping.writeValueAsString(new CalculateBudgetRequest(companyId)),
                ContentType.APPLICATION_JSON));

        post.addHeader(AUTHORIZATION_HEADER, oauthToken);

        try {
            RetryUtils.query(RETRY_POLICY, () -> {
                try (CloseableHttpResponse response = httpClient.execute(post)) {
                    if (response.getStatusLine().getStatusCode() / 100 == 2) {
                        JsonNode node = OM.readTree(response.getEntity().getContent());
                        if (node.get("data") != null) {
                            advertisingInfo.setGeoAdvertisingResponse(OM.convertValue(node.get("data"),
                                    GeoAdvertisingInfo.GeoAdvertisingResponse.class));
                        }

                        return true;
                    } else {
                        String responseString = IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
                        log.error("Error from remote service (" + response.getStatusLine().getStatusCode() + "): " + responseString);
                        throw new RuntimeException("Error from geo adv service");
                    }
                } catch (IOException e) {
                    throw new WebmasterException("Unable to query " + post,
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
                }
            });
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return advertisingInfo;
    }

    private GeoCompanyInfo getCompanyInfo(Long companyId) {
        GeoCompanyInfo companyInfo = new GeoCompanyInfo();
        companyInfo.setPermalink(companyId);

        HttpGet post = new HttpGet(getGeoAdvUrl + GET_COMPANY_METHOD + companyId);

        post.addHeader(AUTHORIZATION_HEADER, oauthToken);

        log.info(getGeoAdvUrl);

        try {
            RetryUtils.query(RETRY_POLICY, () -> {
                try (CloseableHttpResponse response = httpClient.execute(post)) {
                    if (response.getStatusLine().getStatusCode() / 100 == 2) {
                        JsonNode node = OM.readTree(response.getEntity().getContent());
                        if (node.get("data") != null) {
                            companyInfo.setGeoAdvertisingResponse(OM.convertValue(node.get("data"),
                                    GeoCompanyInfo.GeoCompanyResponse.class));
                        }

                        return true;
                    } else {
                        String responseString = IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
                        log.error("Error from remote service (" + response.getStatusLine().getStatusCode() + "): " + responseString);
                        throw new RuntimeException("Error from geo adv service");
                    }
                } catch (IOException e) {
                    throw new WebmasterException("Unable to query " + post,
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
                }
            });
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return companyInfo;
    }

    private GeoAdvertisingInfo empty(Long companyId) {
        final GeoAdvertisingInfo advertisingInfo = new GeoAdvertisingInfo();
        advertisingInfo.setPermalink(companyId);
        return advertisingInfo;
    }
}
