package ru.yandex.webmaster3.storage.host.dao;

import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.regions.data.HostRegion;
import ru.yandex.webmaster3.core.regions.data.HostRegionSourceTypeEnum;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHArrayType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * Created by Oleg Bazdyrev on 25/06/2021.
 */
@Repository
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostRegionsCHDao extends AbstractClickhouseDao {

    public static final CHTable TABLE = CHTable.builder()
            .database(DB_WEBMASTER3_CHECKLIST)
            .name("host_regions_%s")
            .sharded(true)
            .partitionBy("")
            .keyField(F.HOST_ID, CHPrimitiveType.String)
            .field(F.DICTIONARY_REGIONS, new CHArrayType(CHPrimitiveType.Int64))
            .field(F.WEBMASTER_REGIONS, new CHArrayType(CHPrimitiveType.Int64))
            .build();

    @Qualifier("legacyMdbTableStorage")
    private final TableProvider tableProvider;

    public Set<HostRegion> getHostRegions(WebmasterHostId hostId) {
        return getHostRegionsOptional(hostId).orElse(Collections.emptySet());
    }

    public Optional<Set<HostRegion>> getHostRegionsOptional(WebmasterHostId hostId) {
        ClickhouseTableInfo table = tableProvider.getTable(TableType.HOST_REGIONS);
        Where st = QueryBuilder.select(F.DICTIONARY_REGIONS, F.WEBMASTER_REGIONS)
                .from(table.getLocalTableName()).where(QueryBuilder.eq(F.HOST_ID, hostId));

        return getClickhouseServer().queryOne(table.chContext(getClickhouseServer(), hostId), st, chRow -> {
            Set<HostRegion> result = new HashSet<>();
            chRow.getLongListUnsafe(F.DICTIONARY_REGIONS).stream()
                    .map(id -> new HostRegion((int) (id % 1000000), HostRegionSourceTypeEnum.DICTIONARY)).forEach(result::add);
            chRow.getLongListUnsafe(F.WEBMASTER_REGIONS).stream()
                    .map(id -> new HostRegion((int) (id % 1000000), HostRegionSourceTypeEnum.WEBMASTER)).forEach(result::add);
            return result;
        });
    }

    private interface F {
        String HOST_ID = "host_id";
        String DICTIONARY_REGIONS = "dictionary_regions";
        String WEBMASTER_REGIONS = "webmaster_regions";
    }
}
