package ru.yandex.webmaster3.storage.host.dao;

import java.util.Collection;
import java.util.function.Consumer;
import java.util.function.Function;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostGeneration;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author iceflame
 */
@Repository
public class HostsYDao extends AbstractYDao {

    public HostsYDao() {
        super(PREFIX_HOST, "hosts");
    }

    private static final ValueDataMapper<WebmasterHostGeneration> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, WebmasterHostGeneration::getHostId),
            Pair.of(F.ADD_DATE, WebmasterHostGeneration::getAddedDate)
    );
    private static final ValueDataMapper<WebmasterHostId> DELETE_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_ID, Function.identity())
    );
    private static final DataMapper<Pair<WebmasterHostId, DateTime>> PAIR_MAPPER = DataMapper.create(
            F.HOST_ID, F.ADD_DATE, Pair::of
    );

    public void deleteHost(WebmasterHostId hostId) {
        delete().where(F.HOST_ID.eq(hostId)).execute();
    }

    public void deleteHosts(Collection<WebmasterHostId> hostIds) {
        batchDelete(DELETE_VALUE_MAPPER, hostIds).execute();
    }

    public boolean isHostAdded(WebmasterHostId hostId) {
        return select(F.HOST_ID).where(F.HOST_ID.eq(hostId)).queryOne() != null;
    }

    public void addHost(WebmasterHostId hostId, DateTime addedDate) {
        upsert(F.HOST_ID.value(hostId), F.ADD_DATE.value(addedDate)).execute();
    }

    public void addHosts(Collection<WebmasterHostGeneration> hosts) {
        batchInsert(INSERT_VALUE_MAPPER, hosts).execute();
    }

    public void forEach(Consumer<Pair<WebmasterHostId, DateTime>> consumer) {
        streamReader(PAIR_MAPPER, consumer);
    }

    public void forEachHost(Consumer<WebmasterHostId> consumer) {
        streamReader(F.HOST_ID, consumer);
    }

    private interface F {
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<DateTime> ADD_DATE = Fields.jodaDateTimeField("add_date").makeOptional();
    }
}
